/**
 * Washington Theme - Headless SPA
 * Modern, minimalist e-commerce experience
 */

(function() {
    'use strict';

    // Danish number formatting
    function formatPrice(price) {
        if (typeof price === 'string') {
            price = parseFloat(price);
        }
        return price.toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    }

    // API Client
    class ShoporamaAPI {
        constructor(baseUrl = '/api') {
            this.baseUrl = baseUrl;
            this.basketToken = localStorage.getItem('basket_token');
        }

        async request(endpoint, options = {}) {
            try {
                let url = this.baseUrl + endpoint;
                
                // Add basket token to basket endpoints
                if (this.basketToken && endpoint.includes('/basket')) {
                    const separator = endpoint.includes('?') ? '&' : '?';
                    url += `${separator}basket_token=${this.basketToken}`;
                }
                
                const response = await fetch(url, {
                    ...options,
                    credentials: 'include', // Include cookies for session
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        ...options.headers
                    }
                });

                const responseText = await response.text();
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (e) {
                    console.error('Invalid JSON response length:', responseText.length);
                    console.error('First 200 chars:', responseText.substring(0, 200));
                    console.error('Last 200 chars:', responseText.substring(responseText.length - 200));
                    console.error('JSON parse error:', e.message);
                    throw new Error('Ugyldig server respons');
                }
                if (!data.success) throw new Error(data.error?.message || 'API Error');
                return data.data;
            } catch (error) {
                console.error('API Error:', error);
                throw error;
            }
        }

        async getProducts(params = {}) {
            const query = new URLSearchParams(params).toString();
            return this.request(`/products${query ? '?' + query : ''}`);
        }

        async getProduct(id) {
            return this.request(`/products/${id}`);
        }

        async getCategories() {
            return this.request('/categories');
        }

        async getMenus() {
            return this.request('/menus');
        }

        async addToBasket(productId, amount = 1, attributes = {}) {
            const data = await this.request('/basket', {
                method: 'POST',
                body: JSON.stringify({ 
                    product_id: productId, 
                    amount,
                    attributes: Object.keys(attributes).length > 0 ? attributes : undefined
                })
            });
            if (data.token) {
                this.basketToken = data.token;
                localStorage.setItem('basket_token', data.token);
            }
            if (data.basket_id) {
                localStorage.setItem('basket_id', data.basket_id);
            }
            return data;
        }

        async getBasket() {
            return this.request('/basket');
        }

        async removeFromBasket(productId) {
            // Use PUT with amount=0 to remove item
            return this.request(`/basket/${productId}`, {
                method: 'PUT',
                body: JSON.stringify({ amount: 0 })
            });
        }

        async updateBasketItem(productId, amount) {
            return this.request(`/basket/${productId}`, {
                method: 'PUT',
                body: JSON.stringify({ amount })
            });
        }

        async applyVoucher(code) {
            return this.request('/voucher/apply', {
                method: 'POST',
                body: JSON.stringify({ code })
            });
        }

        async removeVoucher() {
            return this.request('/voucher/remove', {
                method: 'POST'
            });
        }

        async search(query) {
            return this.request(`/search?q=${encodeURIComponent(query)}`);
        }
    }

    // Washington App
    class WashingtonApp {
        constructor() {
            this.api = new ShoporamaAPI('/admin/webshop/controllers/api.php');
            this.state = {
                products: [],
                categories: [],
                menus: [],
                cart: { items: [], total: 0 },
                currentView: 'home',
                loading: false,
                searchOpen: false,
                cartOpen: false,
                mobileMenuOpen: false
            };
            this.init();
        }

        async init() {
            this.setupRouter();
            this.render();
            await this.loadInitialData();
            this.handleRoute();
            this.attachGlobalListeners();
            this.attachCartEventListeners();
        }

        setupRouter() {
            window.addEventListener('popstate', () => this.handleRoute());
            
            document.addEventListener('click', (e) => {
                // Handle both links and buttons with data-route
                if (e.target.matches('[data-route]') || e.target.closest('[data-route]')) {
                    e.preventDefault();
                    const element = e.target.closest('[data-route]') || e.target;
                    const route = element.getAttribute('data-route');
                    this.navigate(route);
                }
            });
        }

        navigate(route) {
            history.pushState(null, '', route);
            this.handleRoute();
            window.scrollTo(0, 0);
        }
        
        updatePageTitle(title) {
            const shopName = window.shopConfig.shopName || 'Webshop';
            if (title && title !== shopName) {
                document.title = `${title} - ${shopName}`;
            } else {
                document.title = shopName;
            }
        }

        async handleRoute() {
            // Prevent multiple simultaneous route handling
            if (this.state.handlingRoute) {
                return;
            }
            this.state.handlingRoute = true;
            
            const path = window.location.pathname.toLowerCase();
            
            if (path === '/' || path === '') {
                this.showHome();
            } else if (path === '/basket' || path === '/cart' || path === '/kurv') {
                // Basket/cart should never have pagination - ignore query params
                if (window.location.search) {
                    // Remove query parameters from basket URL
                    history.replaceState(null, '', '/basket');
                }
                this.showCart();
                this.state.handlingRoute = false;
                return; // Important: return early to avoid the resolve check
            } else if (path === '/address') {
                this.showCheckout();
                this.state.checkoutStep = 'address';
                this.state.handlingRoute = false;
            } else if (path === '/shipping') {
                this.showCheckout();
                this.state.checkoutStep = 'shipping';
                this.state.handlingRoute = false;
                return;
            } else if (path === '/search') {
                this.showSearchResults();
                this.state.handlingRoute = false;
                return;
            } else if (path === '/thanks') {
                this.showThanks();
                this.state.handlingRoute = false;
                return;
            } else if (path === '/order') {
                this.showOrder();
                this.state.handlingRoute = false;
                return;
            } else if (path === '/payment-failed' || path === '/payment-cancelled') {
                this.showPaymentFailed();
                this.state.handlingRoute = false;
                return;
            } else if (path.startsWith('/product/')) {
                // Old URL format for backwards compatibility
                const id = path.split('/').pop();
                this.showProduct(id);
            } else if (path.startsWith('/category/')) {
                // Old URL format for backwards compatibility
                const id = path.split('/').pop();
                this.showCategory(id);
            } else {
                // Use the resolve endpoint that follows webshop.php logic
                const rewriteUrl = path.substring(1); // Remove leading slash
                
                // Show loading state while checking
                this.state.loading = true;
                this.render();
                
                try {
                    // Include query parameters for pagination
                    const queryString = window.location.search;
                    const fullUrl = `${this.api.baseUrl}/resolve/${rewriteUrl}${queryString}`;
                    
                    const response = await fetch(fullUrl, {
                        credentials: 'include'
                    });
                    
                    if (response.ok) {
                        const result = await response.json();
                        
                        if (result.success && result.data) {
                            const { type, data, products, pagination } = result.data;
                            
                            // Get current page from URL parameters
                            const urlParams = new URLSearchParams(window.location.search);
                            const currentPage = parseInt(urlParams.get('p') || '0', 10);
                            
                            switch (type) {
                                case 'home':
                                    this.state.loading = false;
                                    this.showHome();
                                    break;
                                    
                                case 'category':
                                    this.state.currentCategory = data;
                                    this.state.currentCategoryId = data.id;
                                    this.state.pagination = pagination || null;
                                    this.state.currentView = 'category';
                                    this.updatePageTitle(data.name);
                                    
                                    // If loading a specific page > 0, load all pages from 0 to currentPage
                                    if (currentPage > 0 && pagination) {
                                        this.loadAllPagesUpTo('category', currentPage);
                                    } else {
                                        this.state.products = products || [];
                                        this.state.currentPage = currentPage;
                                        this.state.firstLoadedPage = currentPage;
                                        this.state.loading = false;
                                        this.render();
                                        this.setupInfiniteScroll();
                                    }
                                    break;
                                    
                                case 'product':
                                    this.state.currentProduct = data;
                                    this.state.currentView = 'product';
                                    this.state.loading = false;
                                    this.updatePageTitle(data.name);
                                    this.render();
                                    break;
                                    
                                case 'page':
                                    this.state.currentPage = data;
                                    this.state.currentView = 'static';
                                    this.state.loading = false;
                                    this.updatePageTitle(data.name || data.title);
                                    this.render();
                                    break;
                                    
                                case 'blog':
                                    this.state.currentBlogPost = data;
                                    this.state.currentView = 'blog';
                                    this.state.loading = false;
                                    this.updatePageTitle(data.title || data.name);
                                    this.render();
                                    break;
                                    
                                case 'landing':
                                    this.state.currentLandingPage = data;
                                    this.state.pagination = pagination || null;
                                    this.state.currentView = 'landing';
                                    this.updatePageTitle(data.name || data.title);
                                    
                                    // If loading a specific page > 0, load all pages from 0 to currentPage
                                    if (currentPage > 0 && pagination) {
                                        this.loadAllPagesUpTo('landing', currentPage);
                                    } else {
                                        this.state.products = products || [];
                                        this.state.currentPage = currentPage;
                                        this.state.firstLoadedPage = currentPage;
                                        this.state.loading = false;
                                        this.render();
                                        this.setupInfiniteScroll();
                                    }
                                    break;
                                    
                                default:
                                    this.state.loading = false;
                                    this.showHome();
                            }
                        } else {
                            // Not found - show 404
                            this.state.loading = false;
                            this.show404();
                        }
                    } else {
                        // Error - show 404
                        this.state.loading = false;
                        this.show404();
                    }
                } catch (e) {
                    console.error('Error resolving URL:', e);
                    this.state.loading = false;
                    
                    // If it's a network error, show a different message
                    if (e instanceof TypeError && e.message.includes('NetworkError')) {
                        console.error('Network error - API might be unreachable');
                        // Still show 404 but with different handling
                    }
                    
                    this.show404();
                }
            }
            
            // Clear the route handling flag
            this.state.handlingRoute = false;
        }

        async loadInitialData() {
            try {
                // Try to load menus first, fallback to categories
                try {
                    const menuData = await this.api.getMenus();
                    this.state.menus = menuData.menus || [];
                } catch (e) {
                    console.error('Failed to load menus:', e);
                    try {
                        const catData = await this.api.getCategories();
                        this.state.categories = catData.categories || [];
                    } catch (e2) {
                    }
                }
                
                // Load basket separately to ensure it doesn't fail everything
                try {
                    const cartData = await this.api.getBasket();
                    this.state.cart = cartData || { items: [], total: 0 };
                } catch (e) {
                    this.state.cart = { items: [], total: 0 };
                }
                
                this.render();
            } catch (error) {
                console.error('Failed to load initial data:', error);
                this.render(); // Still render even if loading fails
            }
        }

        async showHome(page = 0, append = false) {
            this.updatePageTitle('');
            this.state.currentView = 'home';
            if (!append) {
                this.state.loading = true;
                this.state.products = [];
                // Reset firstLoadedPage when starting fresh
                this.state.firstLoadedPage = undefined;
            }
            this.state.currentPage = page;
            this.state.loadingMore = append;
            
            if (!append) {
                this.render();
            } else {
                // Show loading indicator without re-rendering everything
                this.showLoadingIndicator();
            }
            
            try {
                // Get page from URL params
                const urlParams = new URLSearchParams(window.location.search);
                const pageParam = parseInt(urlParams.get('p')) || 0;
                if (pageParam !== page && !append) {
                    page = pageParam;
                    this.state.currentPage = page;
                    
                    // If URL has a page number > 0, load all pages from 0 to that page
                    if (page > 0 && this.state.products.length === 0) {
                        // Show loading state
                        this.state.loading = true;
                        this.render();
                        
                        // Load all pages from 0 to current page
                        for (let i = 0; i <= page; i++) {
                            const offset = i * 20; // Default limit
                            const response = await fetch(this.api.baseUrl + `/frontpage?offset=${offset}`, {
                                credentials: 'include',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'Accept': 'application/json'
                                }
                            });
                            
                            const data = await response.json();
                            if (data.success) {
                                this.state.products = [...this.state.products, ...(data.data.products || [])];
                                this.state.pagination = data.data.pagination || null;
                            }
                        }
                        
                        this.state.loading = false;
                        this.state.currentPage = page;
                        this.state.firstLoadedPage = 0; // We loaded from page 0
                        this.render();
                        this.setupInfiniteScroll();
                        
                        // Scroll to bottom after a short delay to let DOM render
                        setTimeout(() => {
                            window.scrollTo(0, document.body.scrollHeight);
                        }, 100);
                        
                        return;
                    }
                }
                
                // Calculate offset from page number (page is 0-based)
                const limit = this.state.pagination?.limit || 25; // Use admin default
                const offset = page * limit;
                
                // Use the new frontpage endpoint to get frontpage products with pagination
                const response = await fetch(this.api.baseUrl + `/frontpage?offset=${offset}`, {
                    credentials: 'include',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                });
                
                const data = await response.json();
                let newProducts = [];
                
                if (data.success) {
                    newProducts = data.data.products || [];
                    if (append) {
                        // Append to existing products
                        this.state.products = [...this.state.products, ...newProducts];
                    } else {
                        this.state.products = newProducts;
                    }
                    this.state.pagination = data.data.pagination || null;
                } else {
                    // Fallback to getting all products
                    const fallbackData = await this.api.getProducts({ limit: 50, offset: offset });
                    newProducts = fallbackData.products || [];
                    if (append) {
                        this.state.products = [...this.state.products, ...newProducts];
                    } else {
                        this.state.products = newProducts;
                    }
                    this.state.pagination = fallbackData.pagination || null;
                }
                
                // If appending, just add the new products to DOM
                if (append && newProducts.length > 0) {
                    this.appendProductsToDOM(newProducts);
                }
                
            } catch (error) {
                console.error('Failed to load products:', error);
                // Try fallback
                try {
                    const data = await this.api.getProducts({ limit: 50 });
                    if (!append) {
                        this.state.products = data.products || [];
                    }
                    this.state.pagination = data.pagination || null;
                } catch (e) {
                    console.error('Fallback also failed:', e);
                }
            }
            
            this.state.loading = false;
            this.state.loadingMore = false;
            
            if (append) {
                this.hideLoadingIndicator();
            } else {
                this.render();
            }
            
            // Setup infinite scroll for home view
            if (!append && this.state.currentView === 'home') {
                this.setupInfiniteScroll();
            }
        }
        
        showLoadingIndicator() {
            const existingIndicator = document.getElementById('infinite-scroll-loader');
            if (!existingIndicator) {
                const gridContainer = document.querySelector('.grid');
                if (gridContainer) {
                    const loader = document.createElement('div');
                    loader.id = 'infinite-scroll-loader';
                    loader.className = 'w-full flex flex-col items-center justify-center py-12';
                    loader.innerHTML = `
                        <div class="relative">
                            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-charcoal"></div>
                            <div class="absolute inset-0 animate-spin rounded-full h-12 w-12 border-t-2 border-nordic opacity-40" style="animation-duration: 1.5s; animation-direction: reverse;"></div>
                        </div>
                        <p class="text-charcoal/60 mt-4 text-sm animate-pulse">Indlæser flere produkter...</p>
                    `;
                    gridContainer.parentElement.appendChild(loader);
                }
            }
        }
        
        hideLoadingIndicator() {
            const indicator = document.getElementById('infinite-scroll-loader');
            if (indicator) {
                // Fade out animation
                indicator.style.transition = 'opacity 0.3s ease-out';
                indicator.style.opacity = '0';
                setTimeout(() => {
                    indicator.remove();
                }, 300);
            }
        }
        
        appendProductsToDOM(newProducts) {
            const gridContainer = document.querySelector('.grid');
            if (gridContainer && newProducts.length > 0) {
                // Create document fragment for better performance
                const fragment = document.createDocumentFragment();
                
                newProducts.forEach(product => {
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = this.renderProductCard(product);
                    const productElement = tempDiv.firstElementChild;
                    
                    // Attach route listeners directly to new product links
                    const links = productElement.querySelectorAll('a[data-route]');
                    links.forEach(link => {
                        link.addEventListener('click', (e) => {
                            e.preventDefault();
                            const route = link.getAttribute('data-route');
                            if (route) {
                                history.pushState({}, '', route);
                                this.handleRoute();
                            }
                        });
                    });
                    
                    fragment.appendChild(productElement);
                });
                
                gridContainer.appendChild(fragment);
            }
        }

        async showProduct(id) {
            this.state.currentView = 'product';
            this.state.loading = true;
            
            // Remove infinite scroll handler when viewing a product
            if (this.scrollHandler) {
                window.removeEventListener('scroll', this.scrollHandler);
                this.scrollHandler = null;
            }
            
            this.render();
            
            try {
                const product = await this.api.getProduct(id);
                this.state.currentProduct = product;
                this.updatePageTitle(product.name);
                
                // Also load related products
                const data = await this.api.getProducts({ limit: 4 });
                this.state.relatedProducts = data.products || [];
            } catch (error) {
                console.error('Failed to load product:', error);
            }
            
            this.state.loading = false;
            this.render();
        }

        async showCategory(id, page = 0, append = false) {
            this.state.currentView = 'category';
            this.state.currentCategoryId = id;
            if (!append) {
                this.state.loading = true;
                this.state.products = [];
                // Reset firstLoadedPage when starting fresh
                this.state.firstLoadedPage = undefined;
            }
            this.state.currentPage = page;
            this.state.loadingMore = append;
            
            if (!append) {
                this.render();
            } else {
                // Show loading indicator without re-rendering everything
                this.showLoadingIndicator();
            }
            
            try {
                // Get page from URL params
                const urlParams = new URLSearchParams(window.location.search);
                const pageParam = parseInt(urlParams.get('p')) || 0;
                if (pageParam !== page && !append) {
                    page = pageParam;
                    this.state.currentPage = page;
                    
                    // If URL has a page number > 0, load all pages from 0 to that page
                    if (page > 0 && this.state.products.length === 0) {
                        // Show loading state
                        this.state.loading = true;
                        this.render();
                        
                        // Load all pages from 0 to current page
                        for (let i = 0; i <= page; i++) {
                            const offset = i * 20; // Default limit
                            const data = await this.api.getProducts({ 
                                category_id: id, 
                                offset: offset 
                            });
                            
                            if (data.products) {
                                this.state.products = [...this.state.products, ...(data.products || [])];
                                this.state.pagination = data.pagination || null;
                            }
                        }
                        
                        this.state.loading = false;
                        this.state.currentPage = page;
                        this.state.firstLoadedPage = 0; // We loaded from page 0
                        this.render();
                        this.setupInfiniteScroll();
                        
                        // Scroll to bottom after a short delay to let DOM render
                        setTimeout(() => {
                            window.scrollTo(0, document.body.scrollHeight);
                        }, 100);
                        
                        return;
                    }
                }
                
                // Calculate offset from page number using pagination limit if available
                const limit = this.state.pagination?.limit || 25; // Use admin default
                const offset = page * limit;
                
                const data = await this.api.getProducts({ 
                    category_id: id, 
                    offset: offset 
                });
                
                let newProducts = data.products || [];
                
                if (append) {
                    this.state.products = [...this.state.products, ...newProducts];
                    // Append new products to DOM without re-rendering
                    if (newProducts.length > 0) {
                        this.appendProductsToDOM(newProducts);
                    }
                } else {
                    this.state.products = newProducts;
                }
                this.state.pagination = data.pagination || null;
            } catch (error) {
                console.error('Failed to load category:', error);
            }
            
            this.state.loading = false;
            this.state.loadingMore = false;
            
            if (append) {
                this.hideLoadingIndicator();
            } else {
                this.render();
            }
            
            // Setup infinite scroll for category view
            if (!append && this.state.currentView === 'category') {
                this.setupInfiniteScroll();
            }
        }
        
        setupInfiniteScroll() {
            // Remove any existing scroll listener first to prevent duplicates
            if (this.scrollHandler) {
                window.removeEventListener('scroll', this.scrollHandler);
                this.scrollHandler = null;
            }
            
            // Only setup infinite scroll for product listings (home, category, landing pages)
            // NOT for individual product pages
            if (this.state.currentView === 'product' || this.state.currentView === 'static' || 
                this.state.currentView === 'blog' || this.state.currentView === 'cart' || 
                this.state.currentView === 'checkout') {
                return;
            }
            
            // Track the first page we loaded (for reverse scrolling)
            // This should already be set in showHome/showCategory, but as fallback:
            if (this.state.firstLoadedPage === undefined) {
                this.state.firstLoadedPage = this.state.currentPage || 0;
            }
            
            this.scrollHandler = () => {
                // Double-check we're not on a product page
                if (this.state.currentView === 'product') {
                    return;
                }
                
                const scrollPosition = window.innerHeight + window.scrollY;
                const pageHeight = document.documentElement.offsetHeight;
                const scrollTop = window.scrollY;
                const threshold = 200;
                
                // Check if we're near the bottom (load next page)
                if (scrollPosition >= pageHeight - threshold && 
                    !this.state.loadingMore && 
                    this.state.pagination && 
                    this.state.currentPage < this.state.pagination.pages - 1) {
                    
                    // Load next page
                    const nextPage = this.state.currentPage + 1;
                    
                    // Update URL with new page number
                    const url = new URL(window.location);
                    if (nextPage > 0) {
                        url.searchParams.set('p', nextPage);
                    } else {
                        url.searchParams.delete('p');
                    }
                    window.history.replaceState({}, '', url);
                    
                    this.state.loadingMore = true;
                    this.showLoadingIndicator();
                    
                    if (this.state.currentView === 'home') {
                        this.showHome(nextPage, true);
                    } else if (this.state.currentView === 'category' || this.state.currentView === 'landing') {
                        // Use resolve endpoint for consistency
                        this.loadMoreViaResolve(nextPage);
                    }
                }
                
                // Check if we're near the top (load previous page)
                if (scrollTop <= threshold && 
                    !this.state.loadingMore && 
                    this.state.firstLoadedPage > 0) {
                    
                    // Load previous page
                    const prevPage = this.state.firstLoadedPage - 1;
                    this.state.firstLoadedPage = prevPage;
                    
                    if (this.state.currentView === 'home') {
                        this.loadPreviousPage(prevPage);
                    } else if (this.state.currentView === 'category' || this.state.currentView === 'landing') {
                        this.loadPreviousViaResolve(prevPage);
                    }
                }
            };
            
            // Add scroll listener with debounce
            let scrollTimeout;
            window.addEventListener('scroll', () => {
                clearTimeout(scrollTimeout);
                scrollTimeout = setTimeout(this.scrollHandler, 100);
            });
        }
        
        async loadPreviousPage(page) {
            this.state.loadingMore = true;
            this.showLoadingIndicatorTop();
            
            try {
                const offset = page * (this.state.pagination?.limit || 25); // Use admin default limit
                const response = await fetch(this.api.baseUrl + `/frontpage?offset=${offset}`, {
                    credentials: 'include',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                });
                
                const data = await response.json();
                if (data.success && data.data.products) {
                    // Prepend products to the beginning
                    this.prependProductsToDOM(data.data.products);
                    this.state.products = [...data.data.products, ...this.state.products];
                }
            } catch (error) {
                console.error('Failed to load previous page:', error);
            }
            
            this.state.loadingMore = false;
            this.hideLoadingIndicatorTop();
        }
        
        async loadPreviousPageCategory(categoryId, page) {
            this.state.loadingMore = true;
            this.showLoadingIndicatorTop();
            
            try {
                const offset = page * (this.state.pagination?.limit || 25); // Use admin default limit
                const data = await this.api.getProducts({ 
                    category_id: categoryId, 
                    offset: offset 
                });
                
                if (data.products) {
                    // Prepend products to the beginning
                    this.prependProductsToDOM(data.products);
                    this.state.products = [...data.products, ...this.state.products];
                }
            } catch (error) {
                console.error('Failed to load previous page:', error);
            }
            
            this.state.loadingMore = false;
            this.hideLoadingIndicatorTop();
        }

        async loadPreviousPageLanding(rewriteUrl, page) {
            this.state.loadingMore = true;
            this.showLoadingIndicatorTop();
            
            try {
                const response = await fetch(`${this.api.baseUrl}/resolve/${rewriteUrl}?p=${page}`, {
                    credentials: 'include'
                });
                
                const result = await response.json();
                if (result.success && result.data && result.data.products) {
                    // Prepend products to the beginning
                    this.prependProductsToDOM(result.data.products);
                    this.state.products = [...result.data.products, ...this.state.products];
                }
            } catch (error) {
                console.error('Failed to load previous page:', error);
            }
            
            this.state.loadingMore = false;
            this.hideLoadingIndicatorTop();
        }

        async loadMoreViaResolve(page) {
            try {
                const rewriteUrl = window.location.pathname.substring(1); // Remove leading slash
                const response = await fetch(`${this.api.baseUrl}/resolve/${rewriteUrl}?p=${page}`, {
                    credentials: 'include'
                });
                
                if (response.ok) {
                    const result = await response.json();
                    
                    if (result.success && result.data && result.data.products) {
                        // Append products to existing list
                        this.state.products = [...this.state.products, ...result.data.products];
                        this.state.currentPage = page;
                        this.appendProductsToDOM(result.data.products);
                    }
                }
            } catch (error) {
                console.error('Failed to load more products:', error);
            }
            
            this.state.loadingMore = false;
            this.hideLoadingIndicator();
        }

        async loadPreviousViaResolve(page) {
            this.state.loadingMore = true;
            this.showLoadingIndicatorTop();
            
            try {
                const rewriteUrl = window.location.pathname.substring(1); // Remove leading slash
                const response = await fetch(`${this.api.baseUrl}/resolve/${rewriteUrl}?p=${page}`, {
                    credentials: 'include'
                });
                
                if (response.ok) {
                    const result = await response.json();
                    
                    if (result.success && result.data && result.data.products) {
                        // Prepend products to beginning
                        this.state.products = [...result.data.products, ...this.state.products];
                        this.prependProductsToDOM(result.data.products);
                    }
                }
            } catch (error) {
                console.error('Failed to load previous products:', error);
            }
            
            this.state.loadingMore = false;
            this.hideLoadingIndicatorTop();
        }

        async loadAllPagesUpTo(type, targetPage) {
            this.state.loading = true;
            this.state.products = [];
            this.render(); // Show loading state
            
            try {
                const rewriteUrl = window.location.pathname.substring(1);
                const allProducts = [];
                
                // Load all pages from 0 to targetPage
                for (let page = 0; page <= targetPage; page++) {
                    const response = await fetch(`${this.api.baseUrl}/resolve/${rewriteUrl}?p=${page}`, {
                        credentials: 'include'
                    });
                    
                    if (response.ok) {
                        const result = await response.json();
                        if (result.success && result.data && result.data.products) {
                            allProducts.push(...result.data.products);
                        }
                    }
                }
                
                // Set all the loaded products
                this.state.products = allProducts;
                this.state.currentPage = targetPage;
                this.state.firstLoadedPage = 0; // We loaded from page 0
                this.state.loading = false;
                this.render();
                this.setupInfiniteScroll();
                
                // Scroll to the appropriate position (roughly where page targetPage would start)
                if (targetPage > 0) {
                    setTimeout(() => {
                        const productsPerPage = this.state.pagination?.limit || 25;
                        const targetProductIndex = targetPage * productsPerPage;
                        const productCards = document.querySelectorAll('.grid > div');
                        
                        if (productCards[targetProductIndex]) {
                            productCards[targetProductIndex].scrollIntoView({ 
                                behavior: 'smooth',
                                block: 'start'
                            });
                        }
                    }, 100); // Small delay to let DOM render
                }
                
            } catch (error) {
                console.error('Failed to load all pages:', error);
                this.state.loading = false;
                this.render();
            }
        }
        
        showLoadingIndicatorTop() {
            const existingIndicator = document.getElementById('infinite-scroll-loader-top');
            if (!existingIndicator) {
                const gridContainer = document.querySelector('.grid');
                if (gridContainer) {
                    const loader = document.createElement('div');
                    loader.id = 'infinite-scroll-loader-top';
                    loader.className = 'w-full flex flex-col items-center justify-center py-12';
                    loader.innerHTML = `
                        <div class="relative">
                            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-charcoal"></div>
                            <div class="absolute inset-0 animate-spin rounded-full h-12 w-12 border-t-2 border-nordic opacity-40" style="animation-duration: 1.5s; animation-direction: reverse;"></div>
                        </div>
                        <p class="text-charcoal/60 mt-4 text-sm animate-pulse">Indlæser tidligere produkter...</p>
                    `;
                    gridContainer.parentElement.insertBefore(loader, gridContainer);
                }
            }
        }
        
        hideLoadingIndicatorTop() {
            const indicator = document.getElementById('infinite-scroll-loader-top');
            if (indicator) {
                indicator.style.transition = 'opacity 0.3s ease-out';
                indicator.style.opacity = '0';
                setTimeout(() => {
                    indicator.remove();
                }, 300);
            }
        }
        
        prependProductsToDOM(newProducts) {
            const gridContainer = document.querySelector('.grid');
            if (gridContainer && newProducts.length > 0) {
                // Remember scroll position
                const scrollBefore = window.scrollY;
                const heightBefore = document.documentElement.scrollHeight;
                
                // Get first existing child to insert before
                const firstChild = gridContainer.firstChild;
                
                // Add products in reverse order so they appear in correct order
                newProducts.reverse().forEach(product => {
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = this.renderProductCard(product);
                    const productElement = tempDiv.firstElementChild;
                    
                    // Attach route listeners directly to new product links
                    const links = productElement.querySelectorAll('a[data-route]');
                    links.forEach(link => {
                        link.addEventListener('click', (e) => {
                            e.preventDefault();
                            const route = link.getAttribute('data-route');
                            if (route) {
                                history.pushState({}, '', route);
                                this.handleRoute();
                            }
                        });
                    });
                    
                    // Insert at the beginning
                    gridContainer.insertBefore(productElement, gridContainer.firstChild);
                });
                
                // Maintain scroll position
                const heightAfter = document.documentElement.scrollHeight;
                const heightDiff = heightAfter - heightBefore;
                window.scrollTo(0, scrollBefore + heightDiff);
            }
        }

        showCart() {
            this.updatePageTitle('Indkøbskurv');
            this.state.currentView = 'cart';
            this.state.cartOpen = false;
            // Clear pagination state for cart view
            this.state.pagination = null;
            this.render();
        }

        show404() {
            this.updatePageTitle('Side ikke fundet');
            this.state.currentView = '404';
            this.state.cartOpen = false;
            this.state.pagination = null;
            this.render();
        }

        async showLandingPage(rewriteUrl, page = 0, append = false) {
            try {
                const response = await fetch(`${this.api.baseUrl}/resolve/${rewriteUrl}?p=${page}`, {
                    credentials: 'include'
                });
                
                if (response.ok) {
                    const result = await response.json();
                    
                    if (result.success && result.data && result.data.type === 'landing') {
                        const { data, products, pagination } = result.data;
                        
                        if (append) {
                            // Append products
                            this.state.products = [...this.state.products, ...(products || [])];
                            this.appendProductsToDOM(products || []);
                        } else {
                            // Replace products
                            this.state.currentLandingPage = data;
                            this.state.products = products || [];
                            this.state.currentView = 'landing';
                            this.state.loading = false;
                            this.updatePageTitle(data.name || data.title);
                            this.render();
                        }
                        
                        this.state.pagination = pagination || null;
                        this.state.currentPage = page;
                        
                        // Setup infinite scroll for landing page view
                        if (!append && this.state.currentView === 'landing') {
                            this.setupInfiniteScroll();
                        }
                    }
                }
            } catch (error) {
                console.error('Failed to load landing page:', error);
            }
            
            this.state.loading = false;
        }

        async showCheckout() {
            this.updatePageTitle('Checkout');
            // Check if cart has items
            if (!this.state.cart.items || this.state.cart.items.length === 0) {
                this.showNotification('Din kurv er tom', 'error');
                this.navigate('/basket');
                return;
            }
            
            this.state.currentView = 'checkout';
            this.state.cartOpen = false;
            this.state.checkoutStep = 'address';
            this.state.termsAccepted = false; // Reset terms acceptance when starting checkout
            
            // Load checkout data
            try {
                const response = await fetch(`${this.api.baseUrl}/checkout`, {
                    credentials: 'include'
                });
                const data = await response.json();
                
                if (data.success) {
                    this.state.shippingMethods = data.data.shipping_methods || [];
                    this.state.paymentMethods = data.data.payment_methods || [];
                    this.state.countries = data.data.countries || [];
                    this.state.savedOrder = data.data.saved_order || {};
                    this.state.savedDelivery = data.data.saved_delivery || {};
                }
            } catch (error) {
                console.error('Failed to load checkout data:', error);
            }
            
            this.render();
        }

        async showSearchResults() {
            this.state.currentView = 'search';
            this.render();
        }
        
        showThanks() {
            this.updatePageTitle('Tak for din ordre');
            this.state.currentView = 'thanks';
            
            // Get order info from URL parameters
            const params = new URLSearchParams(window.location.search);
            this.state.thanksData = {
                orderId: params.get('order_id'),
                basketId: params.get('basket_id'),
                email: params.get('email')
            };
            
            // Clear cart after successful order
            this.state.cart = { items: [], total: 0, subtotal: 0 };
            this.updateCartCount();
            
            // Clear pending order from session storage
            sessionStorage.removeItem('pending_order');
            
            this.render();
        }
        
        async showOrder() {
            this.updatePageTitle('Ordre');
            this.state.currentView = 'order';
            this.state.loading = true;
            this.state.error = null;
            this.state.order = null;
            
            this.render();
            
            // Get order parameters from URL
            const params = new URLSearchParams(window.location.search);
            const orderId = params.get('order_id');
            const chk = params.get('chk');
            
            if (!orderId || !chk) {
                this.state.error = 'Manglende ordre parametre';
                this.state.loading = false;
                this.render();
                return;
            }
            
            try {
                const response = await fetch(`${this.api.baseUrl}/order?order_id=${orderId}&chk=${chk}`, {
                    credentials: 'include'
                });
                
                const data = await response.json();
                
                if (data.success && data.data && data.data.order) {
                    this.state.order = data.data.order;
                } else {
                    this.state.error = data.error?.message || 'Ordre ikke fundet';
                }
            } catch (err) {
                this.state.error = 'Der opstod en fejl ved indlæsning af ordren';
                console.error('Order loading error:', err);
            }
            
            this.state.loading = false;
            this.render();
        }
        
        showPaymentFailed() {
            this.updatePageTitle('Betaling mislykket');
            this.state.currentView = 'payment-failed';
            
            // Check if we have a pending order in session storage
            const pendingOrder = sessionStorage.getItem('pending_order');
            if (pendingOrder) {
                try {
                    this.state.pendingOrder = JSON.parse(pendingOrder);
                } catch (e) {
                    console.error('Invalid JSON in pending_order:', e);
                    sessionStorage.removeItem('pending_order');
                    this.state.pendingOrder = null;
                }
            }
            
            this.render();
        }

        async addToCart(productId, amount = 1, attributes = {}) {
            try {
                const data = await this.api.addToBasket(productId, amount, attributes);
                this.state.cart = data;
                this.showNotification('Tilføjet til kurven', 'success');
                this.state.cartOpen = true;
                this.render();
            } catch (error) {
                console.error('Failed to add to cart:', error);
                const errorMsg = error.message || 'Kunne ikke tilføje til kurven';
                this.showNotification(errorMsg, 'error');
            }
        }
        
        updateVariantInfo() {
            const product = this.state.currentProduct;
            if (!product || !product.variants) return;
            
            // Get selected attributes from both selects and buttons
            const selectedAttributes = {};
            
            // From select dropdowns
            document.querySelectorAll('select[data-variant-attribute]').forEach(select => {
                const attrId = select.getAttribute('data-variant-attribute');
                const value = select.value;
                if (value) {
                    selectedAttributes[attrId] = value;
                }
            });
            
            // From button groups
            document.querySelectorAll('[data-variant-group]').forEach(group => {
                const attrId = group.getAttribute('data-variant-group');
                const selectedButton = group.querySelector('button[data-selected="true"]');
                if (selectedButton) {
                    const valueId = selectedButton.getAttribute('data-variant-value');
                    selectedAttributes[attrId] = valueId;
                }
            });
            
            // Find matching variant
            const matchingVariant = product.variants.find(variant => {
                const variantAttrs = variant.attributes;
                return Object.keys(selectedAttributes).every(attrId => 
                    variantAttrs[attrId] == selectedAttributes[attrId]
                );
            });
            
            // Update variant info display
            const variantInfo = document.getElementById('variant-info');
            const variantPrice = document.getElementById('variant-price');
            const variantStock = document.getElementById('variant-stock');
            const addToCartBtn = document.querySelector('[data-add-to-cart]');
            
            if (matchingVariant) {
                let hasInfoToShow = false;
                
                // Update price display
                if (matchingVariant.price && matchingVariant.price > 0) {
                    // Variant has its own price - update main price display
                    const mainPriceContainer = document.querySelector('.mb-8');
                    if (mainPriceContainer) {
                        // Build the new price HTML
                        let priceHTML = `<span class="text-2xl text-charcoal">${formatPrice(matchingVariant.price)} ${window.shopConfig.currency}</span>`;
                        
                        // If there's an old_price (normal price) and it's higher than current price, show it as crossed out
                        if (matchingVariant.old_price && matchingVariant.old_price > matchingVariant.price) {
                            priceHTML += `<span class="text-xl text-charcoal/40 line-through ml-3">${formatPrice(matchingVariant.old_price)} ${window.shopConfig.currency}</span>`;
                        }
                        
                        mainPriceContainer.innerHTML = priceHTML;
                    }
                    
                    // Don't show price in variant info box - it's already shown in main price
                    if (variantPrice) {
                        variantPrice.textContent = '';
                    }
                } else {
                    // No variant-specific price, keep original product price
                    if (variantPrice) {
                        variantPrice.textContent = '';
                    }
                }
                
                // Update stock display
                if (variantStock) {
                    if (matchingVariant.stock > 0) {
                        variantStock.textContent = `På lager: ${matchingVariant.stock} stk`;
                        variantStock.classList.remove('text-red-600');
                        variantStock.classList.add('text-green-600');
                        hasInfoToShow = true;
                    } else {
                        // Check if product allows negative stock
                        if (product.allow_negative_stock) {
                            // Don't show stock info when allowing negative stock and out of stock
                            variantStock.textContent = '';
                            variantStock.classList.remove('text-red-600', 'text-green-600');
                        } else {
                            variantStock.textContent = 'Udsolgt';
                            variantStock.classList.remove('text-green-600');
                            variantStock.classList.add('text-red-600');
                            hasInfoToShow = true;
                        }
                    }
                }
                
                // Show/hide the info box based on whether there's info to display
                if (variantInfo) {
                    if (hasInfoToShow) {
                        variantInfo.classList.remove('hidden');
                    } else {
                        variantInfo.classList.add('hidden');
                    }
                }
                
                // Enable/disable add to cart button based on stock
                if (addToCartBtn) {
                    if (matchingVariant.stock > 0 || product.allow_negative_stock) {
                        addToCartBtn.disabled = false;
                        addToCartBtn.classList.remove('opacity-50', 'cursor-not-allowed');
                        addToCartBtn.textContent = 'TILFØJ TIL KURV';
                    } else {
                        addToCartBtn.disabled = true;
                        addToCartBtn.classList.add('opacity-50', 'cursor-not-allowed');
                        addToCartBtn.textContent = 'UDSOLGT';
                    }
                }
            } else {
                // No matching variant found or not all attributes selected
                if (variantInfo) {
                    variantInfo.classList.add('hidden');
                }
                
                // Reset price to original display (could be range or fixed price)
                const product = this.state.currentProduct;
                if (product) {
                    const mainPriceContainer = document.querySelector('.mb-8');
                    if (mainPriceContainer) {
                        // Recalculate price display logic (same as in renderProductDetail)
                        let displayPrice = product.price;
                        let displayOldPrice = product.old_price;
                        let hasPriceRange = false;
                        let priceRangeMin = null;
                        let priceRangeMax = null;
                        
                        if (product.variants && product.variants.length > 0) {
                            const variantsWithPrice = product.variants.filter(v => v.price && v.price > 0);
                            if (variantsWithPrice.length > 0) {
                                const prices = variantsWithPrice.map(v => parseFloat(v.price));
                                priceRangeMin = Math.min(...prices);
                                priceRangeMax = Math.max(...prices);
                                
                                if (priceRangeMin !== priceRangeMax) {
                                    hasPriceRange = true;
                                    displayPrice = null;
                                    displayOldPrice = null;
                                } else if ((!displayPrice || displayPrice === 0)) {
                                    displayPrice = priceRangeMin;
                                    displayOldPrice = variantsWithPrice[0].old_price;
                                }
                            }
                        }
                        
                        let priceHTML = '';
                        if (hasPriceRange) {
                            priceHTML = `<span class="text-2xl text-charcoal">${formatPrice(priceRangeMin)} - ${formatPrice(priceRangeMax)} ${window.shopConfig.currency}</span>
                                        <div class="text-sm text-charcoal/60 mt-1">Prisen varierer baseret på valgte optioner</div>`;
                        } else if (displayPrice) {
                            priceHTML = `<span class="text-2xl text-charcoal">${formatPrice(displayPrice)} ${window.shopConfig.currency}</span>`;
                            if (displayOldPrice) {
                                priceHTML += `<span class="text-xl text-charcoal/40 line-through ml-3">${formatPrice(displayOldPrice)} ${window.shopConfig.currency}</span>`;
                            }
                        } else {
                            priceHTML = `<span class="text-xl text-charcoal/60">Vælg optioner for at se pris</span>`;
                        }
                        
                        mainPriceContainer.innerHTML = priceHTML;
                    }
                }
                
                // Reset add to cart button
                if (addToCartBtn) {
                    addToCartBtn.disabled = false;
                    addToCartBtn.classList.remove('opacity-50', 'cursor-not-allowed');
                    addToCartBtn.textContent = 'TILFØJ TIL KURV';
                }
            }
        }
        
        async showTermsModal() {
            // Fetch terms if not already loaded
            if (!this.state.termsContent) {
                try {
                    const response = await fetch(`${this.api.baseUrl}/settings`, {
                        credentials: 'include'
                    });
                    const data = await response.json();
                    
                    if (data.success && data.data.has_terms) {
                        this.state.termsContent = data.data.terms;
                    } else {
                        this.state.termsContent = '<p>Denne shop har ikke nogen handelsbetingelser.</p>';
                    }
                } catch (err) {
                    console.error('Failed to load terms:', err);
                    this.state.termsContent = '<p>Kunne ikke indlæse handelsbetingelser.</p>';
                }
            }
            
            // Create proper styled modal with Tailwind classes
            const modal = document.createElement('div');
            modal.style.cssText = 'position: fixed; inset: 0; z-index: 9999; display: flex; align-items: center; justify-content: center; padding: 1rem;';
            
            // Create backdrop
            const backdrop = document.createElement('div');
            backdrop.style.cssText = 'position: fixed; inset: 0; background-color: rgba(0, 0, 0, 0.3);';
            backdrop.onclick = () => modal.remove();
            
            // Create modal content
            const modalContent = document.createElement('div');
            modalContent.style.cssText = `
                position: relative;
                background: white;
                border-radius: 0.5rem;
                box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
                max-width: 32rem;
                width: 100%;
                max-height: 70vh;
                display: flex;
                flex-direction: column;
            `;
            
            // Header
            const header = document.createElement('div');
            header.style.cssText = 'padding: 1.5rem; border-bottom: 1px solid #e5e7eb; display: flex; justify-content: space-between; align-items: center;';
            
            const title = document.createElement('h3');
            title.style.cssText = 'font-size: 1.125rem; font-weight: 600; color: #111827;';
            title.textContent = 'Handelsbetingelser';
            
            const closeBtn = document.createElement('button');
            closeBtn.style.cssText = 'padding: 0.25rem; border-radius: 0.25rem; color: #6b7280; cursor: pointer;';
            closeBtn.innerHTML = `
                <svg style="width: 1.25rem; height: 1.25rem;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            `;
            closeBtn.onclick = () => modal.remove();
            
            header.appendChild(title);
            header.appendChild(closeBtn);
            
            // Content
            const content = document.createElement('div');
            content.style.cssText = 'padding: 1.5rem; overflow-y: auto; flex: 1;';
            content.innerHTML = `
                <div style="font-size: 0.875rem; line-height: 1.5; color: #4b5563;">
                    ${this.state.termsContent}
                </div>
            `;
            
            // Footer
            const footer = document.createElement('div');
            footer.style.cssText = 'padding: 1rem 1.5rem; border-top: 1px solid #e5e7eb; display: flex; justify-content: flex-end;';
            
            const footerBtn = document.createElement('button');
            footerBtn.style.cssText = 'padding: 0.5rem 1.5rem; background: #111827; color: white; border-radius: 0.25rem; font-size: 0.875rem; font-weight: 500; cursor: pointer;';
            footerBtn.textContent = 'Luk';
            footerBtn.onclick = () => modal.remove();
            footerBtn.onmouseover = () => footerBtn.style.background = '#1f2937';
            footerBtn.onmouseout = () => footerBtn.style.background = '#111827';
            
            footer.appendChild(footerBtn);
            
            // Assemble modal
            modalContent.appendChild(header);
            modalContent.appendChild(content);
            modalContent.appendChild(footer);
            modal.appendChild(backdrop);
            modal.appendChild(modalContent);
            
            // Add animation styles if not already present
            if (!document.getElementById('modal-animations')) {
                const style = document.createElement('style');
                style.id = 'modal-animations';
                style.innerHTML = `
                    @keyframes fadeIn {
                        from { opacity: 0; }
                        to { opacity: 1; }
                    }
                    @keyframes slideUp {
                        from { 
                            opacity: 0;
                            transform: translateY(20px);
                        }
                        to { 
                            opacity: 1;
                            transform: translateY(0);
                        }
                    }
                `;
                document.head.appendChild(style);
            }
            
            document.body.appendChild(modal);
        }
        
        updateCompleteOrderButton() {
            const button = document.querySelector('[data-complete-order]');
            if (button) {
                if (this.state.termsAccepted) {
                    button.disabled = false;
                    button.classList.remove('opacity-50', 'cursor-not-allowed');
                } else {
                    button.disabled = true;
                    button.classList.add('opacity-50', 'cursor-not-allowed');
                }
            }
        }
        
        changeProductImage(direction) {
            const product = this.state.currentProduct;
            if (!product || !product.images || product.images.length <= 1) return;
            
            const mainImage = document.getElementById('main-product-image');
            if (!mainImage) return;
            
            const currentIndex = parseInt(mainImage.getAttribute('data-current-index') || 0);
            const newIndex = (currentIndex + direction + product.images.length) % product.images.length;
            
            this.setProductImage(newIndex);
        }
        
        setProductImage(index) {
            const product = this.state.currentProduct;
            if (!product || !product.images || !product.images[index]) return;
            
            const mainImage = document.getElementById('main-product-image');
            if (!mainImage) return;
            
            const newImage = product.images[index];
            const newSrc = newImage.formats?.large || newImage.url;
            
            // Update main image
            mainImage.src = newSrc;
            mainImage.setAttribute('data-current-index', index);
            
            // Update indicators
            document.querySelectorAll('[data-image-indicator]').forEach(indicator => {
                const indicatorIndex = parseInt(indicator.getAttribute('data-image-indicator'));
                if (indicatorIndex === index) {
                    indicator.classList.add('bg-charcoal', 'w-6');
                    indicator.classList.remove('bg-charcoal/30');
                } else {
                    indicator.classList.remove('bg-charcoal', 'w-6');
                    indicator.classList.add('bg-charcoal/30');
                }
            });
            
            // Add active class to thumbnail
            document.querySelectorAll('[data-thumbnail-index]').forEach(thumb => {
                const thumbIndex = parseInt(thumb.getAttribute('data-thumbnail-index'));
                if (thumbIndex === index) {
                    thumb.classList.add('ring-2', 'ring-nordic');
                } else {
                    thumb.classList.remove('ring-2', 'ring-nordic');
                }
            });
        }

        async removeFromCart(productId) {
            try {
                const data = await this.api.removeFromBasket(productId);
                this.state.cart = data;
                
                // Simply reload the cart page to show the updated state
                if (this.state.currentView === 'cart' || window.location.pathname.toLowerCase() === '/basket') {
                    this.showCart();
                } else if (this.state.cartOpen) {
                    this.updateCartSidebarContent();
                }
                
                // Update cart count in header
                this.updateCartCount();
            } catch (error) {
                console.error('Failed to remove from cart:', error);
            }
        }
        
        updateCartCount() {
            const cartCount = document.getElementById('cart-count');
            if (cartCount) {
                const items = this.state.cart.items || [];
                const totalCount = items.reduce((sum, item) => sum + item.amount, 0);
                cartCount.textContent = totalCount;
            }
        }
        
        updateCartSidebarContent() {
            const cartContent = document.querySelector('#cart-sidebar .overflow-y-auto');
            if (cartContent) {
                const items = this.state.cart.items || [];
                cartContent.innerHTML = items.length === 0 ? `
                    <p class="text-center text-charcoal/60 py-8">Din kurv er tom</p>
                ` : `
                    <div class="space-y-4">
                        ${items.map(item => `
                            <div class="flex space-x-3">
                                <img src="${item.image?.formats?.thumbnail || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"%3E%3Crect width="80" height="80" fill="%23f5f5dc"/%3E%3C/svg%3E'}" 
                                     class="w-20 h-20 object-contain p-1">
                                <div class="flex-1">
                                    <h4 class="font-medium mb-1">${item.name}</h4>
                                    ${this.formatVariantInfo(item)}
                                    <p class="text-sm text-charcoal/60 mb-2">${formatPrice(item.price)} ${window.shopConfig.currency}</p>
                                    <div class="flex items-center space-x-2">
                                        <button data-update-cart="${item.product_id}" data-amount="${item.amount - 1}"
                                                class="text-charcoal/40 hover:text-charcoal">-</button>
                                        <span class="px-2">${item.amount}</span>
                                        <button data-update-cart="${item.product_id}" data-amount="${item.amount + 1}"
                                                class="text-charcoal/40 hover:text-charcoal">+</button>
                                    </div>
                                </div>
                                <button data-remove-from-cart="${item.product_id}"
                                        class="text-charcoal/40 hover:text-charcoal">
                                    <svg class="w-5 h-5 pointer-events-none" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6 18L18 6M6 6l12 12"></path>
                                    </svg>
                                </button>
                            </div>
                        `).join('')}
                    </div>
                `;
                
                // Update total in footer
                const cartTotal = document.querySelector('#cart-sidebar .border-t.p-6');
                if (cartTotal) {
                    if (items.length > 0) {
                        const subtotal = this.state.cart.subtotal || 0;
                        const shipping = this.state.cart.shipping || 0;
                        const total = this.state.cart.total || 0;
                        
                        cartTotal.innerHTML = `
                            <div class="space-y-2 mb-4">
                                <div class="flex justify-between text-sm">
                                    <span>Subtotal</span>
                                    <span>${formatPrice(subtotal)} ${window.shopConfig.currency}</span>
                                </div>
                                ${shipping > 0 ? `
                                    <div class="flex justify-between text-sm">
                                        <span>Fragt</span>
                                        <span>${formatPrice(shipping)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                <div class="flex justify-between font-serif text-lg pt-2 border-t">
                                    <span>Total</span>
                                    <span>${formatPrice(total)} ${window.shopConfig.currency}</span>
                                </div>
                            </div>
                            <a href="/basket" data-route="/basket" 
                               class="block w-full bg-charcoal text-white text-center py-3 hover:bg-nordic">
                                Gå til kurv
                            </a>
                        `;
                        cartTotal.style.display = 'block';
                    } else {
                        cartTotal.style.display = 'none';
                    }
                }
                
                // Event delegation handles this automatically, no need to reattach
            }
        }
        
        attachCartEventListeners() {
            // Only attach once - check if already attached
            if (this.cartEventListenersAttached) {
                return;
            }
            
            // Use arrow function to preserve context
            this.cartDelegatedHandler = (e) => {
                // Check if target or any parent has cart action attributes
                const updateButton = e.target.closest('[data-update-cart]');
                const removeButton = e.target.closest('[data-remove-from-cart]');
                
                if (updateButton) {
                    e.preventDefault();
                    e.stopPropagation();
                    const productId = updateButton.getAttribute('data-update-cart');
                    const amount = parseInt(updateButton.getAttribute('data-amount'));
                    
                    if (this.updateCartItem) {
                        this.updateCartItem(productId, amount);
                    } else {
                        console.error('updateCartItem method not found!');
                    }
                    return;
                }
                
                if (removeButton) {
                    e.preventDefault();
                    e.stopPropagation();
                    const productId = removeButton.getAttribute('data-remove-from-cart');
                    this.removeFromCart(productId);
                    return;
                }
            };
            
            // Add delegated event listener to document with capture phase
            document.addEventListener('click', this.cartDelegatedHandler, true);
            this.cartEventListenersAttached = true;
        }

        async updateCartItem(productId, amount) {
            
            if (amount <= 0) {
                return this.removeFromCart(productId);
            }
            
            try {
                const data = await this.api.updateBasketItem(productId, amount);
                this.state.cart = data;
                
                // Simply reload the cart page to show the updated state
                if (this.state.currentView === 'cart' || window.location.pathname.toLowerCase() === '/basket') {
                        this.showCart();
                } else if (this.state.cartOpen) {
                    this.updateCartSidebarContent();
                }
                
                this.updateCartCount();
                
            } catch (error) {
                console.error('Failed to update cart - Full error:', error);
                console.error('Error stack:', error.stack);
            }
        }

        async search(query) {
            if (!query) {
                this.state.searchResults = [];
                this.state.searchQuery = '';
                this.updateSearchResults();
                return;
            }
            
            this.state.isSearching = true;
            this.state.searchQuery = query;
            this.updateSearchResults();
            
            try {
                const data = await this.api.search(query);
                this.state.searchResults = data.products || [];
                // Don't navigate away, keep overlay open to show results
            } catch (error) {
                console.error('Search failed:', error);
                this.state.searchResults = [];
            }
            
            this.state.isSearching = false;
            this.updateSearchResults();
        }
        
        updateSearchResults() {
            // Only update the search results div, not the entire overlay
            const resultsDiv = document.getElementById('search-results');
            if (!resultsDiv) return;
            
            const searchResults = this.state.searchResults || [];
            const isSearching = this.state.isSearching || false;
            
            resultsDiv.innerHTML = isSearching ? `
                <p class="text-charcoal/40">Søger...</p>
            ` : searchResults.length > 0 ? `
                <div class="grid grid-cols-4 md:grid-cols-6 lg:grid-cols-8 gap-4">
                    ${searchResults.map(product => {
                        const productUrl = product.rewrite_url || `product/${product.id}`;
                        const imageUrl = product.images?.[0]?.formats?.medium || 
                                       product.images?.[0]?.url || 
                                       'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="400" height="400" viewBox="0 0 400 400"%3E%3Crect width="400" height="400" fill="%23f5f5dc"/%3E%3C/svg%3E';
                        return `
                            <a href="/${productUrl}" data-route="/${productUrl}" 
                               class="group cursor-pointer">
                                <div class="aspect-square bg-light-gray mb-3 overflow-hidden">
                                    <img src="${imageUrl}" 
                                         alt="${product.name}"
                                         class="w-full h-full object-contain p-3 group-hover:scale-105 transition-transform duration-300"
                                         loading="lazy">
                                </div>
                                <h3 class="text-sm font-medium group-hover:underline line-clamp-2">${product.name}</h3>
                                <p class="text-sm font-semibold mt-1">${formatPrice(product.price)} ${window.shopConfig.currency}</p>
                            </a>
                        `;
                    }).join('')}
                </div>
            ` : this.state.searchQuery ? `
                <p class="text-charcoal/40">Ingen produkter fundet</p>
            ` : `
                <p class="text-charcoal/40">Begynd at skrive for at søge...</p>
            `;
            
            // Re-attach route event listeners for the new product links
            resultsDiv.querySelectorAll('[data-route]').forEach(link => {
                link.addEventListener('click', (e) => {
                    e.preventDefault();
                    const route = link.getAttribute('data-route');
                    // Close search overlay first
                    this.state.searchOpen = false;
                    this.state.searchResults = [];
                    this.state.searchQuery = '';
                    // Navigate to product
                    this.navigate(route);
                });
            });
        }

        showNotification(message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = `fixed bottom-8 left-1/2 transform -translate-x-1/2 px-8 py-4 rounded-lg text-white z-50 fade-in ${
                type === 'error' ? 'bg-red-600' : 'bg-sage'
            }`;
            notification.innerHTML = `
                <div class="flex items-center space-x-3">
                    ${type === 'success' ? `
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                        </svg>
                    ` : `
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    `}
                    <span class="font-medium">${message}</span>
                </div>
            `;
            document.body.appendChild(notification);
            
            setTimeout(() => notification.remove(), 3000);
        }
        
        showVoucherMessage(message, type = 'info') {
            const messageDiv = document.getElementById('voucher-message');
            if (!messageDiv) return;
            
            messageDiv.textContent = message;
            messageDiv.className = `mt-2 text-sm ${
                type === 'success' ? 'text-green-600' : 
                type === 'error' ? 'text-red-600' : 
                'text-gray-600'
            }`;
            messageDiv.classList.remove('hidden');
            
            // Auto hide after 5 seconds
            setTimeout(() => {
                messageDiv.classList.add('hidden');
            }, 5000);
        }

        showNotification(message, type = 'info', duration = 4000) {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-sm ${
                type === 'error' ? 'bg-red-100 border border-red-300 text-red-800' : 
                type === 'success' ? 'bg-green-100 border border-green-300 text-green-800' : 
                'bg-blue-100 border border-blue-300 text-blue-800'
            } transition-all duration-300 transform translate-x-full opacity-0`;
            
            notification.textContent = message;
            document.body.appendChild(notification);
            
            // Animate in
            setTimeout(() => {
                notification.classList.remove('translate-x-full', 'opacity-0');
            }, 10);
            
            // Auto-remove
            setTimeout(() => {
                notification.classList.add('translate-x-full', 'opacity-0');
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, duration);
        }

        formatVariantInfo(item) {
            if (!item.attributes || Object.keys(item.attributes).length === 0) {
                return '';
            }

            // Find the product to get attribute names
            const product = this.state.products.find(p => p.id == item.product_id);
            if (!product || !product.attributes) return '';

            const variantParts = [];
            Object.entries(item.attributes).forEach(([attrId, valueId]) => {
                const attribute = product.attributes.find(attr => attr.id == attrId);
                if (attribute) {
                    const value = attribute.values.find(val => val.id == valueId);
                    if (value) {
                        variantParts.push(`${attribute.name}: ${value.value}`);
                    }
                }
            });

            return variantParts.length > 0 ? `<div class="text-xs text-charcoal/50 mt-1">${variantParts.join(', ')}</div>` : '';
        }

        render() {
            const app = document.getElementById('app');
            if (!app) return;
            
            app.innerHTML = `
                ${this.renderHeader()}
                <main class="flex-1">
                    ${this.state.loading ? this.renderLoading() : this.renderContent()}
                </main>
                ${this.renderFooter()}
                ${this.renderCartSidebar()}
                ${this.renderSearchOverlay()}
            `;
            
            this.attachEventListeners();
            // Don't re-attach cart listeners, they use delegation
        }

        renderHeader() {
            const scrolled = window.scrollY > 50;
            const itemCount = this.state.cart.items?.reduce((sum, item) => sum + item.amount, 0) || 0;
            const cartTotal = this.state.cart.total || 0;
            const menus = this.state.menus || [];
            const topMenu = menus.find(menu => menu.location === 'top') || menus[0];
            
            return `
                <header class="sticky top-0 z-40 bg-white border-b border-gray-100 transition-all ${scrolled ? 'shadow-sm' : ''}">
                    <div class="px-3 sm:px-6 lg:px-12">
                        <div class="flex justify-between items-center h-12 sm:h-16">
                            <!-- Mobile Menu Button -->
                            <button id="mobile-menu-toggle" class="sm:hidden p-2 text-gray-700 hover:text-gray-900 focus:outline-none">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16" />
                                </svg>
                            </button>
                            
                            <!-- Logo -->
                            <a href="/" data-route="/" class="group flex-shrink-0 flex items-center space-x-3">
                                ${window.shopConfig.logoUrl ? `
                                    <img src="${window.shopConfig.logoUrl}" alt="${window.shopConfig.shopName}" 
                                         class="h-8 sm:h-10 lg:h-12 w-auto object-contain">
                                ` : ''}
                                <h1 class="text-base sm:text-2xl lg:text-3xl font-light tracking-wide sm:tracking-widest text-gray-900 hover:text-gray-700 transition-colors"
                                    style="font-family: 'Playfair Display', Georgia, serif;">
                                    ${window.shopConfig.shopName.toUpperCase()}
                                </h1>
                            </a>
                            
                            <!-- Actions - Compact on mobile -->
                            <div class="flex items-center space-x-2 sm:space-x-6 lg:space-x-8">
                                <!-- Search - Icon only on mobile -->
                                <button id="search-toggle" class="p-2 sm:p-0 text-gray-600 hover:text-gray-900 transition-colors">
                                    <svg class="h-5 w-5 sm:hidden" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                                    </svg>
                                    <span class="hidden sm:inline text-sm tracking-wide">SØG</span>
                                </button>
                                
                                <!-- Cart - Optimized for mobile -->
                                <button id="cart-toggle" class="relative flex items-center">
                                    <!-- Mobile: Icon with badge only -->
                                    <div class="sm:hidden p-2 relative">
                                        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-charcoal" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                                        </svg>
                                        ${itemCount > 0 ? `
                                            <span class="absolute -top-0.5 -right-0.5 bg-nordic text-white text-xs font-bold rounded-full h-5 w-5 flex items-center justify-center">
                                                ${itemCount}
                                            </span>
                                        ` : ''}
                                    </div>
                                    
                                    <!-- Desktop: Full button -->
                                    <div class="hidden sm:flex relative bg-charcoal text-white px-3 lg:px-4 py-2 rounded-md hover:bg-nordic transition-all">
                                        <div class="flex items-center space-x-2">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                                            </svg>
                                            <span class="text-sm font-medium tracking-wide">
                                                ${itemCount > 0 ? `${formatPrice(cartTotal)} DKK` : 'KURV'}
                                            </span>
                                        </div>
                                        ${itemCount > 0 ? `
                                            <span class="absolute -top-2 -right-2 bg-nordic text-white text-xs font-bold rounded-full h-6 w-6 flex items-center justify-center shadow-md">
                                                ${itemCount}
                                            </span>
                                        ` : ''}
                                    </div>
                                </button>
                            </div>
                        </div>
                        
                        <!-- Desktop Navigation Menu - Below logo -->
                        ${topMenu && topMenu.items && topMenu.items.length > 0 ? `
                            <nav class="border-t border-gray-100" style="${window.innerWidth < 640 ? 'display: none;' : 'display: block;'}">
                                <ul class="flex justify-center items-center space-x-6 py-3">
                                    ${topMenu.items.map(item => `
                                        <li class="relative group">
                                            <a href="${item.url}" data-route="${item.url}"
                                               class="text-sm tracking-wide text-gray-700 hover:text-gray-900 py-2 px-3 transition-colors whitespace-nowrap">
                                                ${item.title.toUpperCase()}
                                            </a>
                                            ${item.has_children ? this.renderSubMenu(item.children) : ''}
                                        </li>
                                    `).join('')}
                                </ul>
                            </nav>
                        ` : ''}
                        
                        <!-- Mobile Menu Overlay -->
                        <div id="mobile-menu" class="sm:hidden fixed inset-0 bg-black bg-opacity-50 z-50 ${this.state.mobileMenuOpen ? '' : 'hidden'}">
                            <div class="absolute left-0 top-0 h-full w-80 bg-white shadow-xl" style="transition: transform 0.3s ease-in-out;">
                                <div class="bg-gray-900 text-white p-5">
                                    <div class="flex justify-between items-center">
                                        <h2 class="text-xl font-light tracking-wider">MENU</h2>
                                        <button id="mobile-menu-close" class="p-1 text-white hover:text-gray-300">
                                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6 18L18 6M6 6l12 12" />
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                                <nav class="overflow-y-auto h-full pb-20">
                                    ${topMenu && topMenu.items && topMenu.items.length > 0 ? `
                                        <ul>
                                            ${topMenu.items.map((item, index) => `
                                                <li>
                                                    ${item.has_children ? `
                                                        <div>
                                                            <a href="${item.url}" data-route="${item.url}"
                                                               class="block px-6 py-4 text-gray-800 mobile-menu-link font-medium text-base border-b border-gray-100 hover:bg-gray-50 transition-colors">
                                                                ${item.title}
                                                            </a>
                                                            <ul>
                                                                ${item.children.map(child => `
                                                                    <li>
                                                                        <a href="${child.url}" data-route="${child.url}"
                                                                           class="block text-gray-600 mobile-menu-link hover:bg-gray-50 transition-colors border-b border-gray-100 text-sm" style="padding: 12px 24px 12px 50px; font-size: 14px;">
                                                                            ${child.title}
                                                                        </a>
                                                                    </li>
                                                                `).join('')}
                                                            </ul>
                                                        </div>
                                                    ` : `
                                                        <a href="${item.url}" data-route="${item.url}"
                                                           class="block px-6 py-4 text-gray-800 mobile-menu-link font-medium text-base border-b border-gray-100 hover:bg-gray-50 transition-colors">
                                                            ${item.title}
                                                        </a>
                                                    `}
                                                </li>
                                            `).join('')}
                                        </ul>
                                    ` : ''}
                                </nav>
                            </div>
                        </div>
                    </div>
                </header>
            `;
        }
        
        
        renderSubMenu(children) {
            return `
                <div class="submenu-dropdown absolute left-0 bg-white border border-gray-200 rounded-lg shadow-lg opacity-0 invisible transition-all duration-200 z-50" style="width: 220px; top: 100%; margin-top: 8px;">
                    <ul class="py-2">
                        ${children.map(child => `
                            <li>
                                <a href="${child.url}" data-route="${child.url}"
                                   class="block px-3 py-3 text-sm text-gray-700 hover:bg-gray-50 transition-colors" style="white-space: nowrap;">
                                    ${child.title}
                                </a>
                            </li>
                        `).join('')}
                    </ul>
                </div>
            `;
        }

        renderContent() {
            switch(this.state.currentView) {
                case 'home':
                    return this.renderHome();
                case 'product':
                    return this.renderProductDetail();
                case 'category':
                    return this.renderCategory();
                case 'cart':
                    return this.renderCartPage();
                case 'checkout':
                    return this.renderCheckout();
                case 'search':
                    return this.renderSearchResults();
                case 'thanks':
                    return this.renderThanks();
                case 'order':
                    return this.renderOrder();
                case 'payment-failed':
                    return this.renderPaymentFailed();
                case 'static':
                    return this.renderStaticPage();
                case 'landing':
                    return this.renderLandingPage();
                case '404':
                    return this.render404();
                default:
                    return this.renderHome();
            }
        }
        
        renderStaticPage() {
            const page = this.state.currentPage;
            if (!page) return '<div class="p-8 text-center">Side ikke fundet</div>';
            
            return `
                <div class="px-6 lg:px-12 py-12">
                    <div class="max-w-4xl mx-auto">
                        <h1 class="font-serif text-4xl mb-8">${page.title}</h1>
                        <div class="prose prose-lg max-w-none">
                            ${page.content}
                        </div>
                    </div>
                </div>
            `;
        }
        
        renderLandingPage() {
            const page = this.state.currentLandingPage;
            if (!page) return '<div class="p-8 text-center">Side ikke fundet</div>';
            
            const products = this.state.products || [];
            
            return `
                <div class="px-6 lg:px-12 py-12">
                    <h1 class="font-serif text-3xl text-center mb-12">${page.name}</h1>
                    
                    ${page.description ? `
                        <div class="max-w-4xl mx-auto mb-12 text-center text-charcoal/80">
                            ${page.description}
                        </div>
                    ` : ''}
                    
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 2xl:grid-cols-6 gap-8">
                        ${products.map(product => this.renderProductCard(product)).join('')}
                    </div>
                    
                    ${products.length === 0 && !this.state.loading ? `
                        <p class="text-center text-charcoal/60 py-8">Ingen produkter fundet</p>
                    ` : ''}
                </div>
            `;
        }

        render404() {
            return `
                <div class="min-h-screen flex items-center justify-center bg-gradient-to-br from-snow to-beige px-6 lg:px-12">
                    <div class="max-w-2xl mx-auto text-center">
                        <!-- 404 Icon/Illustration -->
                        <div class="mb-8">
                            <div class="relative inline-block">
                                <div class="text-9xl font-serif text-charcoal/10 select-none">404</div>
                                <div class="absolute inset-0 flex items-center justify-center">
                                    <svg class="w-16 h-16 text-nordic" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" 
                                              d="M9.172 16.172a4 4 0 015.656 0M9 12h6m-6-4h6m2 5.291A7.962 7.962 0 0112 15c-2.137 0-4.146-.832-5.636-2.364M6.343 7.343A7.963 7.963 0 0112 5c4.418 0 8 3.582 8 8a7.96 7.96 0 01-2.343 5.657M6.343 18.657A7.963 7.963 0 015 14c0-1.01.156-1.985.445-2.897"/>
                                    </svg>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Title and Message -->
                        <h1 class="font-serif text-4xl md:text-5xl text-charcoal mb-4">
                            Ups! Side ikke fundet
                        </h1>
                        <p class="text-charcoal/70 mb-8 text-lg md:text-xl leading-relaxed max-w-lg mx-auto">
                            Den side du leder efter eksisterer ikke eller er blevet flyttet. 
                            Lad os hjælpe dig med at finde vej tilbage.
                        </p>
                        
                        <!-- Action Buttons -->
                        <div class="flex flex-col sm:flex-row gap-4 justify-center items-center">
                            <a href="/" data-route="/" 
                               class="btn btn-primary text-lg px-8 py-4 shadow-lg hover:shadow-xl transform hover:-translate-y-1 transition-all duration-200">
                                🏠 Gå til forsiden
                            </a>
                            <button onclick="history.back()" 
                                    class="btn btn-secondary text-lg px-8 py-4 shadow-md hover:shadow-lg transform hover:-translate-y-1 transition-all duration-200">
                                ← Gå tilbage
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }

        renderHome() {
            return `
                <!-- Products Grid -->
                <section class="px-6 lg:px-12 py-8">
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 2xl:grid-cols-6 gap-6">
                        ${this.state.products.map(product => this.renderProductCard(product)).join('')}
                    </div>
                </section>
            `;
        }
        
        renderPagination() {
            // Never show pagination on cart or checkout pages
            const currentPath = window.location.pathname || '/';
            if (currentPath === '/basket' || currentPath === '/address' || currentPath === '/shipping' || currentPath === '/kurv') {
                return '';
            }
            
            if (!this.state.pagination || this.state.pagination.pages <= 1) {
                return '';
            }
            
            const { pages, current_page, limit, offset } = this.state.pagination;
            
            let paginationHTML = '<nav class="flex justify-center items-center space-x-2 mt-12">';
            
            // Previous button
            if (current_page > 1) {
                const prevPage = current_page - 2;
                paginationHTML += `
                    <a href="${currentPath}?p=${prevPage}" 
                       data-page="${prevPage}"
                       class="px-4 py-2 text-charcoal hover:bg-light-gray rounded-md transition">
                        ‹ Forrige
                    </a>
                `;
            }
            
            // Page numbers - show max 5 pages
            let startPage = Math.max(1, current_page - 2);
            let endPage = Math.min(pages, startPage + 4);
            if (endPage - startPage < 4) {
                startPage = Math.max(1, endPage - 4);
            }
            
            for (let i = startPage; i <= endPage; i++) {
                const pageIndex = i - 1;
                const activeClass = i === current_page ? 'bg-charcoal text-white' : 'text-charcoal hover:bg-light-gray';
                paginationHTML += `
                    <a href="${currentPath}?p=${pageIndex}" 
                       data-page="${pageIndex}"
                       class="px-4 py-2 ${activeClass} rounded-md transition">
                        ${i}
                    </a>
                `;
            }
            
            // Next button
            if (current_page < pages) {
                const nextPage = current_page;
                paginationHTML += `
                    <a href="${currentPath}?p=${nextPage}" 
                       data-page="${nextPage}"
                       class="px-4 py-2 text-charcoal hover:bg-light-gray rounded-md transition">
                        Næste ›
                    </a>
                `;
            }
            
            paginationHTML += '</nav>';
            
            // Add event listeners after render
            setTimeout(() => {
                document.querySelectorAll('[data-page]').forEach(link => {
                    link.addEventListener('click', (e) => {
                        e.preventDefault();
                        const page = parseInt(link.dataset.page);
                        // Update URL and reload with new page
                        const url = new URL(window.location);
                        url.searchParams.set('p', page);
                        window.history.pushState({}, '', url);
                        this.showHome(page);
                    });
                });
            }, 0);
            
            return paginationHTML;
        }

        renderProductCard(product) {
            // Determine the display price for the card
            let displayPrice = product.price;
            
            // If product price is 0 and there are variants with prices, show "Fra" with lowest price
            let pricePrefix = '';
            if ((!displayPrice || displayPrice === 0) && product.variants && product.variants.length > 0) {
                const variantsWithPrice = product.variants.filter(v => v.price && v.price > 0);
                if (variantsWithPrice.length > 0) {
                    // Find the variant with the lowest price
                    const lowestPriceVariant = variantsWithPrice.reduce((min, v) => 
                        v.price < min.price ? v : min
                    );
                    displayPrice = lowestPriceVariant.price;
                    // Show "Fra" to indicate this is a starting price
                    if (variantsWithPrice.length > 1) {
                        pricePrefix = 'Fra ';
                    }
                }
            }
            
            const imageUrl = product.images?.[0]?.url || 
                           product.images?.[0]?.formats?.medium || 
                           'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="400" height="400" viewBox="0 0 400 400"%3E%3Crect width="400" height="400" fill="%23f5f5dc"/%3E%3Ctext x="50%25" y="50%25" font-family="serif" font-size="20" fill="%23333" text-anchor="middle" dy=".3em"%3ENo Image%3C/text%3E%3C/svg%3E';
            
            const productUrl = product.rewrite_url || `product/${product.id}`;
            
            return `
                <div class="group cursor-pointer fade-in">
                    <a href="/${productUrl}" data-route="/${productUrl}" class="block">
                        <div class="aspect-square overflow-hidden bg-light-gray mb-4">
                            <img src="${imageUrl}" 
                                 alt="${product.name}"
                                 loading="lazy"
                                 class="w-full h-full object-contain p-2 group-hover:scale-105 transition-transform duration-500"
                                 onerror="this.src='data:image/svg+xml,%3Csvg xmlns=\\'http://www.w3.org/2000/svg\\' width=\\'400\\' height=\\'400\\' viewBox=\\'0 0 400 400\\'%3E%3Crect width=\\'400\\' height=\\'400\\' fill=\\'%23f5f5dc\\'/%3E%3C/svg%3E'">
                        </div>
                    </a>
                    
                    <div class="text-center">
                        <h3 class="font-serif text-lg mb-1">
                            <a href="/${productUrl}" data-route="/${productUrl}" 
                               class="text-charcoal hover:text-sage">
                                ${product.name}
                            </a>
                        </h3>
                        <p class="text-charcoal/60 mb-3">${pricePrefix}${formatPrice(displayPrice)} ${window.shopConfig.currency}</p>
                        
                        ${product.variants && product.variants.length > 0 ? `
                            <a href="/${productUrl}" data-route="/${productUrl}"
                               class="text-sm text-nordic hover:text-charcoal font-medium tracking-wide">
                                VÆLG VARIANT
                            </a>
                        ` : `
                            <button data-add-to-cart="${product.id}"
                                    class="text-sm text-nordic hover:text-charcoal font-medium tracking-wide">
                                TILFØJ TIL KURV
                            </button>
                        `}
                    </div>
                </div>
            `;
        }

        renderProductVariants(product) {
            if (!product.attributes || product.attributes.length === 0) {
                return '';
            }
            
            // Filter only variant attributes
            const variantAttributes = product.attributes.filter(attr => attr.is_variant);
            
            if (variantAttributes.length === 0) {
                return '';
            }
            
            let html = '<div class="space-y-6 mb-6">';
            
            variantAttributes.forEach(attr => {
                const shouldUseFancySelector = attr.values.length <= 7;
                
                if (shouldUseFancySelector) {
                    // Fancy button selector for 7 or fewer options
                    html += `
                        <div>
                            <label class="block text-sm font-medium text-charcoal mb-3">${attr.name}</label>
                            <div class="grid grid-cols-2 sm:grid-cols-3 gap-3" data-variant-group="${attr.id}">
                                ${attr.values.map(val => {
                                    // Find the variant with this attribute value to get its price and stock
                                    let variant = null;
                                    let isAvailable = true;
                                    let priceText = '';
                                    let stockText = '';
                                    
                                    if (product.variants) {
                                        variant = product.variants.find(v => 
                                            v.attributes && v.attributes[attr.id] == val.id
                                        );
                                        if (variant) {
                                            isAvailable = variant.stock > 0 || product.allow_negative_stock;
                                            if (variant.price) {
                                                priceText = formatPrice(variant.price);
                                                if (variant.old_price && variant.old_price > variant.price) {
                                                    priceText = `<span class="line-through text-xs opacity-60">${formatPrice(variant.old_price)}</span> ${priceText}`;
                                                }
                                            }
                                            if (variant.stock <= 0 && !product.allow_negative_stock) {
                                                stockText = 'Udsolgt';
                                            }
                                        }
                                    }
                                    
                                    const disabledClass = !isAvailable ? 'opacity-50 cursor-not-allowed' : 'cursor-pointer hover:border-nordic hover:bg-nordic/5';
                                    const disabledAttr = !isAvailable ? 'disabled' : '';
                                    
                                    return `
                                        <button type="button"
                                                data-variant-attribute="${attr.id}"
                                                data-variant-value="${val.id}"
                                                ${disabledAttr}
                                                class="relative p-3 border-2 border-charcoal/20 rounded-lg text-center transition-all ${disabledClass} variant-option">
                                            <div class="font-medium text-charcoal">${val.value}</div>
                                            ${priceText ? `<div class="text-xs text-charcoal/70 mt-1">${priceText} ${window.shopConfig.currency}</div>` : ''}
                                            ${stockText ? `<div class="text-xs text-red-600 mt-1">${stockText}</div>` : ''}
                                            ${!isAvailable ? '<div class="absolute inset-0 bg-white/50 rounded-lg"></div>' : ''}
                                        </button>
                                    `;
                                }).join('')}
                            </div>
                        </div>
                    `;
                } else {
                    // Traditional select dropdown for more than 7 options
                    html += `
                        <div>
                            <label class="block text-sm font-medium text-charcoal mb-2">${attr.name}</label>
                            <select data-variant-attribute="${attr.id}" 
                                    class="w-full px-4 py-2 border border-charcoal/20 focus:border-nordic focus:outline-none rounded">
                                <option value="">Vælg ${attr.name}</option>
                                ${attr.values.map(val => {
                                    // Find the variant with this attribute value to get its price
                                    let priceText = val.value;
                                    let disabled = '';
                                    if (product.variants) {
                                        const variant = product.variants.find(v => 
                                            v.attributes && v.attributes[attr.id] == val.id
                                        );
                                        if (variant) {
                                            if (variant.price) {
                                                priceText += ` - ${formatPrice(variant.price)} ${window.shopConfig.currency}`;
                                                if (variant.old_price && variant.old_price > variant.price) {
                                                    priceText += ` (før ${formatPrice(variant.old_price)} ${window.shopConfig.currency})`;
                                                }
                                            }
                                            if (variant.stock <= 0 && !product.allow_negative_stock) {
                                                priceText += ' - Udsolgt';
                                                disabled = 'disabled';
                                            }
                                        }
                                    }
                                    return `<option value="${val.id}" ${disabled}>${priceText}</option>`;
                                }).join('')}
                            </select>
                        </div>
                    `;
                }
            });
            
            // Show variant information if variants exist
            if (product.variants && product.variants.length > 0) {
                html += `
                    <div id="variant-info" class="hidden mt-4 p-4 bg-light-gray rounded-lg">
                        <div id="variant-price" class="text-lg font-medium text-charcoal"></div>
                        <div id="variant-stock" class="text-sm text-charcoal/60 mt-1"></div>
                    </div>
                `;
            }
            
            html += '</div>';
            
            return html;
        }

        renderProductDetail() {
            const product = this.state.currentProduct;
            if (!product) return '<div class="p-8 text-center">Produkt ikke fundet</div>';
            
            // Determine the display price
            let displayPrice = product.price;
            let displayOldPrice = product.old_price;
            let hasPriceRange = false;
            let priceRangeMin = null;
            let priceRangeMax = null;
            
            // Check if product has variants with different prices
            if (product.variants && product.variants.length > 0) {
                const variantsWithPrice = product.variants.filter(v => v.price && v.price > 0);
                if (variantsWithPrice.length > 0) {
                    const prices = variantsWithPrice.map(v => parseFloat(v.price));
                    priceRangeMin = Math.min(...prices);
                    priceRangeMax = Math.max(...prices);
                    
                    // If variants have different prices, show price range instead of fixed price
                    if (priceRangeMin !== priceRangeMax) {
                        hasPriceRange = true;
                        displayPrice = null; // Don't show a fixed price
                        displayOldPrice = null;
                    } else if ((!displayPrice || displayPrice === 0)) {
                        // All variants have the same price, use that price
                        displayPrice = priceRangeMin;
                        displayOldPrice = variantsWithPrice[0].old_price;
                    }
                }
            }
            
            const mainImage = product.images?.[0]?.formats?.large || 
                            product.images?.[0]?.url || 
                            'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="800" height="800" viewBox="0 0 800 800"%3E%3Crect width="800" height="800" fill="%23f5f5dc"/%3E%3C/svg%3E';
            
            return `
                <div class="py-12">
                    <div class="max-w-[1400px] mx-auto px-6 lg:px-12">
                        <div class="grid grid-cols-1 lg:grid-cols-2 gap-12">
                            <!-- Images -->
                            <div class="fade-in">
                                <div class="aspect-square overflow-hidden bg-light-gray mb-4 relative group" id="main-image-container">
                                    <img src="${mainImage}" 
                                         alt="${product.name}"
                                         class="w-full h-full object-contain p-4"
                                         id="main-product-image"
                                         data-current-index="0">
                                    
                                    ${product.images?.length > 1 ? `
                                        <!-- Navigation arrows -->
                                        <button onclick="window.washingtonApp.changeProductImage(-1)" 
                                                class="absolute left-4 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white p-2 rounded-full opacity-0 group-hover:opacity-100 transition-opacity">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                                            </svg>
                                        </button>
                                        <button onclick="window.washingtonApp.changeProductImage(1)" 
                                                class="absolute right-4 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white p-2 rounded-full opacity-0 group-hover:opacity-100 transition-opacity">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                                            </svg>
                                        </button>
                                        
                                        <!-- Image indicators -->
                                        <div class="absolute bottom-4 left-1/2 -translate-x-1/2 flex space-x-2">
                                            ${product.images.map((img, index) => `
                                                <button onclick="window.washingtonApp.setProductImage(${index})"
                                                        class="w-2 h-2 rounded-full transition-all ${index === 0 ? 'bg-charcoal w-6' : 'bg-charcoal/30'}"
                                                        data-image-indicator="${index}"></button>
                                            `).join('')}
                                        </div>
                                    ` : ''}
                                </div>
                                
                                ${product.images?.length > 1 ? `
                                    <div class="grid grid-cols-4 gap-2">
                                        ${product.images.slice(0, 8).map((img, index) => `
                                            <div class="aspect-square overflow-hidden bg-light-gray">
                                                <img src="${img.formats?.thumbnail || img.thumbnail || img.url}" 
                                                     class="w-full h-full object-contain p-2 cursor-pointer hover:opacity-75"
                                                     onclick="window.washingtonApp.setProductImage(${index})"
                                                     data-thumbnail-index="${index}">
                                            </div>
                                        `).join('')}
                                    </div>
                                ` : ''}
                            </div>
                            
                            <!-- Product Info -->
                            <div class="fade-in" style="animation-delay: 0.1s">
                                <h1 class="font-serif text-4xl text-charcoal mb-4">${product.name}</h1>
                                
                                ${product.categories && product.categories.length > 0 ? `
                                    <div class="mb-4">
                                        <span class="text-sm text-charcoal/60">Kategorier: </span>
                                        ${product.categories.map(cat => `<a href="${cat.is_front ? '/' : '/' + cat.rewrite_url}" data-route="${cat.is_front ? '/' : '/' + cat.rewrite_url}" class="text-sm text-nordic hover:text-charcoal hover:underline">${cat.name}</a>`).join(', ')}
                                    </div>
                                ` : ''}
                                
                                <div class="mb-8">
                                    ${hasPriceRange ? `
                                        <span class="text-2xl text-charcoal">${formatPrice(priceRangeMin)} - ${formatPrice(priceRangeMax)} ${window.shopConfig.currency}</span>
                                        <div class="text-sm text-charcoal/60 mt-1">Prisen varierer baseret på valgte optioner</div>
                                    ` : displayPrice ? `
                                        <span class="text-2xl text-charcoal">${formatPrice(displayPrice)} ${window.shopConfig.currency}</span>
                                        ${displayOldPrice ? `
                                            <span class="text-xl text-charcoal/40 line-through ml-3">${formatPrice(displayOldPrice)} ${window.shopConfig.currency}</span>
                                        ` : ''}
                                    ` : `
                                        <span class="text-xl text-charcoal/60">Vælg optioner for at se pris</span>
                                    `}
                                </div>
                                
                                <div class="prose prose-lg text-charcoal/80 mb-8">
                                    ${product.description || '<p>Ingen beskrivelse tilgængelig.</p>'}
                                </div>
                                
                                ${this.renderProductVariants(product)}
                                
                                <div class="space-y-4">
                                    <button data-add-to-cart="${product.id}"
                                            class="w-full bg-charcoal text-white py-4 hover:bg-nordic font-medium tracking-wide">
                                        TILFØJ TIL KURV
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Related Products -->
                        ${this.state.relatedProducts?.length > 0 ? `
                            <div class="mt-20">
                                <h2 class="font-serif text-2xl text-center mb-12">Relaterede Produkter</h2>
                                <div class="grid grid-cols-2 md:grid-cols-4 gap-6">
                                    ${this.state.relatedProducts.map(p => this.renderProductCard(p)).join('')}
                                </div>
                            </div>
                        ` : ''}
                    </div>
                </div>
            `;
        }

        renderCategory() {
            const category = this.state.currentCategory;
            const categoryName = category ? category.name : 'Kollektion';
            
            return `
                <div class="px-6 lg:px-12 py-12">
                    <h1 class="font-serif text-3xl text-center mb-12">${categoryName}</h1>
                    
                    ${category && category.description ? `
                        <div class="max-w-4xl mx-auto mb-12 text-center text-charcoal/80">
                            ${category.description}
                        </div>
                    ` : ''}
                    
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 2xl:grid-cols-6 gap-8">
                        ${this.state.products.map(product => this.renderProductCard(product)).join('')}
                    </div>
                    
                    ${this.state.products.length === 0 && !this.state.loading ? `
                        <p class="text-center text-charcoal/60 py-8">Ingen produkter fundet i denne kategori</p>
                    ` : ''}
                </div>
            `;
        }

        renderPaymentFailed() {
            const pendingOrder = this.state.pendingOrder;
            
            return `
                <div class="px-6 lg:px-12 py-20 text-center">
                    <div class="max-w-2xl mx-auto">
                        <div class="mb-8">
                            <svg class="w-20 h-20 mx-auto text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        
                        <h1 class="font-serif text-3xl mb-4">Betaling mislykket</h1>
                        
                        <p class="text-xl text-charcoal/60 mb-8">
                            Din betaling kunne ikke gennemføres. Din ordre er gemt, og du kan prøve igen.
                        </p>
                        
                        <div class="space-y-4">
                            ${pendingOrder ? `
                                <button onclick="window.location.href='${pendingOrder.payment_url}'" 
                                        class="inline-block bg-charcoal text-white px-8 py-3 hover:bg-nordic">
                                    Prøv igen
                                </button>
                            ` : ''}
                            
                            <a href="/address" data-route="/address" 
                               class="inline-block border border-charcoal text-charcoal px-8 py-3 hover:bg-light-gray">
                                Tilbage til checkout
                            </a>
                            
                            <a href="/" data-route="/" 
                               class="block text-charcoal/60 hover:text-charcoal mt-4">
                                Fortsæt med at handle
                            </a>
                        </div>
                    </div>
                </div>
            `;
        }
        
        renderThanks() {
            const { orderId, email } = this.state.thanksData || {};
            
            return `
                <div class="px-6 lg:px-12 py-20 text-center">
                    <div class="max-w-2xl mx-auto">
                        <div class="mb-8">
                            <svg class="w-20 h-20 mx-auto text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        
                        <h1 class="font-serif text-3xl mb-4">Tak for din ordre!</h1>
                        
                        <p class="text-xl text-charcoal/60 mb-2">
                            Din ordre er blevet modtaget og behandles nu.
                        </p>
                        
                        ${email ? `
                            <p class="text-charcoal/60 mb-8">
                                Vi har sendt en ordrebekræftelse til <strong>${email}</strong>
                            </p>
                        ` : ''}
                        
                        ${orderId ? `
                            <p class="text-sm text-charcoal/40 mb-8">
                                Ordrenummer: ${orderId}
                            </p>
                        ` : ''}
                        
                        <div class="space-y-4">
                            <a href="/" data-route="/" 
                               class="inline-block bg-charcoal text-white px-8 py-3 hover:bg-nordic">
                                Fortsæt med at handle
                            </a>
                        </div>
                    </div>
                </div>
            `;
        }
        
        renderOrder() {
            if (this.state.error) {
                return `
                    <div class="px-6 lg:px-12 py-20 text-center">
                        <div class="max-w-2xl mx-auto">
                            <h1 class="font-serif text-3xl mb-4 text-red-600">Ordre ikke fundet</h1>
                            <p class="text-xl text-charcoal/60 mb-8">${this.state.error}</p>
                            <a href="/" data-route="/" 
                               class="inline-block bg-charcoal text-white px-8 py-3 hover:bg-nordic">
                                Tilbage til forsiden
                            </a>
                        </div>
                    </div>
                `;
            }
            
            if (!this.state.order) {
                return `
                    <div class="px-6 lg:px-12 py-20 text-center">
                        <div class="animate-spin rounded-full h-16 w-16 border-b-2 border-charcoal mx-auto mb-4"></div>
                        <p class="text-charcoal/60">Indlæser ordre...</p>
                    </div>
                `;
            }
            
            const order = this.state.order;
            
            return `
                <div class="px-4 sm:px-6 lg:px-12 py-8 max-w-4xl mx-auto">
                    <!-- Order Header -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
                        <div class="text-center border-b border-gray-200 pb-6 mb-6">
                            <h1 class="font-serif text-3xl text-charcoal mb-2">
                                Ordre #${order.order_id}
                            </h1>
                            ${order.date ? `<p class="text-charcoal/60">${new Date(order.date).toLocaleDateString('da-DK')}</p>` : ''}
                            <div class="mt-4 flex justify-center space-x-4">
                                <span class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm">${order.status}</span>
                                ${order.payment_status ? `<span class="px-3 py-1 bg-green-100 text-green-800 rounded-full text-sm">${order.payment_status}</span>` : ''}
                            </div>
                        </div>

                        <!-- Order Details Grid -->
                        <div class="grid md:grid-cols-2 gap-6">
                            <!-- Billing Address -->
                            <div class="bg-gray-50 p-4 rounded-lg">
                                <h3 class="font-semibold text-charcoal mb-3">Faktureringsadresse</h3>
                                <div class="text-sm text-charcoal/80">
                                    <div>${order.billing_address.name}</div>
                                    ${order.billing_address.company ? `<div>${order.billing_address.company}</div>` : ''}
                                    <div>${order.billing_address.address}</div>
                                    <div>${order.billing_address.zipcode} ${order.billing_address.city}</div>
                                    <div>${order.billing_address.country}</div>
                                </div>
                            </div>

                            <!-- Shipping Address -->
                            <div class="bg-gray-50 p-4 rounded-lg">
                                <h3 class="font-semibold text-charcoal mb-3">Leveringsadresse</h3>
                                <div class="text-sm text-charcoal/80">
                                    ${order.pickup_address && order.pickup_address.trim() ? `
                                        <div><strong>Pakkeshop:</strong></div>
                                        <div>${order.pickup_address}</div>
                                    ` : `
                                        <div>${order.shipping_address.name}</div>
                                        ${order.shipping_address.company ? `<div>${order.shipping_address.company}</div>` : ''}
                                        <div>${order.shipping_address.address}</div>
                                        <div>${order.shipping_address.zipcode} ${order.shipping_address.city}</div>
                                        <div>${order.shipping_address.country}</div>
                                    `}
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Products -->
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
                        <h3 class="font-serif text-xl text-charcoal mb-4">Produkter</h3>
                        <div class="space-y-4">
                            ${order.products.map(product => `
                                <div class="flex items-center p-4 border border-gray-200 rounded-lg">
                                    ${product.image ? `
                                        <img src="${product.image.url}" 
                                             alt="${product.image.alt}" 
                                             class="w-16 h-16 object-contain bg-white border border-gray-200 rounded mr-4">
                                    ` : `
                                        <div class="w-16 h-16 bg-gray-200 border border-gray-200 rounded mr-4 flex items-center justify-center">
                                            <span class="text-gray-400 text-xs">Ingen billede</span>
                                        </div>
                                    `}
                                    
                                    <div class="flex-1">
                                        <div class="font-medium text-charcoal">${product.name}</div>
                                        <div class="text-sm text-charcoal/60 mt-1">
                                            ${product.quantity} stk. × ${formatPrice(product.price)} DKK = ${formatPrice(product.total)} DKK
                                        </div>
                                        ${product.attributes && product.attributes.length > 0 ? `
                                            <div class="text-xs text-charcoal/50 mt-1">
                                                ${product.attributes.map(attr => `${attr.name}: ${attr.val}`).join(', ')}
                                            </div>
                                        ` : ''}
                                    </div>
                                </div>
                            `).join('')}
                        </div>
                    </div>

                    <!-- Order Totals -->
                    <div class="bg-charcoal text-white rounded-lg p-6 mb-6">
                        <div class="space-y-2">
                            <div class="flex justify-between">
                                <span>Subtotal:</span>
                                <span>${formatPrice(order.totals.subtotal)} DKK</span>
                            </div>
                            ${parseFloat(order.totals.shipping) > 0 ? `
                                <div class="flex justify-between">
                                    <span>Fragt (${order.shipping.method}):</span>
                                    <span>${formatPrice(order.totals.shipping)} DKK</span>
                                </div>
                            ` : ''}
                            ${parseFloat(order.totals.vat) > 0 ? `
                                <div class="flex justify-between">
                                    <span>Moms:</span>
                                    <span>${formatPrice(order.totals.vat)} DKK</span>
                                </div>
                            ` : ''}
                            <div class="flex justify-between text-lg font-semibold border-t border-gray-600 pt-2 mt-3">
                                <span>Total:</span>
                                <span>${formatPrice(order.totals.total)} DKK</span>
                            </div>
                        </div>
                    </div>

                    <!-- Shipping & Payment Info -->
                    <div class="grid md:grid-cols-2 gap-6">
                        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                            <h3 class="font-semibold text-charcoal mb-3">Forsendelse</h3>
                            <div class="text-sm text-charcoal/80">
                                <div><strong>Metode:</strong> ${order.shipping.method}</div>
                                ${order.shipping.tracking_number ? `
                                    <div><strong>Track & Trace:</strong> ${order.shipping.tracking_number}</div>
                                ` : ''}
                            </div>
                        </div>

                        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                            <h3 class="font-semibold text-charcoal mb-3">Betaling</h3>
                            <div class="text-sm text-charcoal/80">
                                <div><strong>Metode:</strong> ${order.payment.method}</div>
                                ${order.payment.transaction_id ? `
                                    <div><strong>Transaktions ID:</strong> ${order.payment.transaction_id}</div>
                                ` : ''}
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        renderSearchResults() {
            return `
                <div class="px-6 lg:px-12 py-12">
                    <h1 class="font-serif text-3xl text-center mb-4">Søgeresultater</h1>
                    <p class="text-center text-charcoal/60 mb-12">
                        ${this.state.products.length} produkter fundet${this.state.searchQuery ? ` for "${this.state.searchQuery}"` : ''}
                    </p>
                    
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 2xl:grid-cols-6 gap-8">
                        ${this.state.products.map(product => this.renderProductCard(product)).join('')}
                    </div>
                </div>
            `;
        }

        renderCheckout() {
            const step = this.state.checkoutStep || 'address';
            
            return `
                <div class="px-4 sm:px-6 lg:px-12 py-6 sm:py-12">
                    <h1 class="font-serif text-2xl sm:text-3xl mb-6 sm:mb-8">Checkout</h1>
                    
                    <!-- Progress indicator - Responsive -->
                    <div class="mb-8 sm:mb-12">
                        <!-- Mobile: Simplified progress -->
                        <div class="sm:hidden">
                            <div class="flex justify-between items-center">
                                <div class="${step === 'address' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                            rounded-full w-8 h-8 text-xs flex items-center justify-center">1</div>
                                <div class="flex-1 h-0.5 bg-light-gray mx-2"></div>
                                <div class="${step === 'shipping' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                            rounded-full w-8 h-8 text-xs flex items-center justify-center">2</div>
                                <div class="flex-1 h-0.5 bg-light-gray mx-2"></div>
                                <div class="${step === 'payment' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                            rounded-full w-8 h-8 text-xs flex items-center justify-center">3</div>
                            </div>
                            <div class="flex justify-between mt-2 text-xs">
                                <span>Adresse</span>
                                <span>Fragt</span>
                                <span>Betaling</span>
                            </div>
                        </div>
                        
                        <!-- Desktop: Full progress -->
                        <div class="hidden sm:flex items-center justify-center">
                            <div class="flex items-center space-x-4">
                                <div class="${step === 'address' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                            rounded-full w-10 h-10 flex items-center justify-center">1</div>
                                <span class="text-sm">Leveringsadresse</span>
                                
                                <div class="w-16 h-0.5 bg-light-gray"></div>
                                
                                <div class="${step === 'shipping' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                            rounded-full w-10 h-10 flex items-center justify-center">2</div>
                                <span class="text-sm">Forsendelse</span>
                                
                                <div class="w-16 h-0.5 bg-light-gray"></div>
                            
                            <div class="${step === 'payment' ? 'bg-charcoal text-white' : 'bg-light-gray text-charcoal'} 
                                        rounded-full w-10 h-10 flex items-center justify-center">3</div>
                            <span class="text-sm">Betaling</span>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-12">
                        <div class="lg:col-span-2">
                            ${this.renderCheckoutStep(step)}
                        </div>
                        
                        <!-- Order summary sidebar -->
                        <div class="bg-beige p-8 h-fit">
                            <h2 class="font-serif text-xl mb-6">Ordreoversigt</h2>
                            
                            <div class="space-y-2 mb-6 pb-6 border-b border-charcoal/10">
                                ${(this.state.cart.items || []).map(item => {
                                    const variantInfo = this.formatVariantInfo(item).replace(/class="[^"]*"/g, 'class="text-xs text-charcoal/40"');
                                    return `
                                        <div class="flex justify-between text-sm">
                                            <div class="flex-1">
                                                <div>${item.name} x ${item.amount}</div>
                                                ${variantInfo}
                                            </div>
                                            <span>${formatPrice(item.subtotal)} ${window.shopConfig.currency}</span>
                                        </div>
                                    `;
                                }).join('')}
                            </div>
                            
                            <div class="space-y-3 mb-6 pb-6 border-b border-charcoal/10">
                                <div class="flex justify-between">
                                    <span>Subtotal</span>
                                    <span>${formatPrice(this.state.cart.subtotal)} ${window.shopConfig.currency}</span>
                                </div>
                                ${this.state.cart.campaign && this.state.cart.campaign.discount > 0 ? `
                                    <div class="flex justify-between text-green-600">
                                        <span>Kampagne rabat</span>
                                        <span>-${formatPrice(this.state.cart.campaign.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                ${this.state.cart.voucher && this.state.cart.voucher.discount > 0 ? `
                                    <div class="flex justify-between text-green-600">
                                        <span>Rabat (${this.state.cart.voucher.code})</span>
                                        <span>-${formatPrice(this.state.cart.voucher.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                <div class="flex justify-between">
                                    <span>Levering</span>
                                    <span>${this.state.cart.shipping > 0 ? formatPrice(this.state.cart.shipping) + ' ' + window.shopConfig.currency : 'Beregnes ved kassen'}</span>
                                </div>
                            </div>
                            
                            <div class="flex justify-between text-xl font-medium">
                                <span>Total</span>
                                <span>${formatPrice(this.state.cart.total)} ${window.shopConfig.currency}</span>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        renderCheckoutStep(step) {
            switch(step) {
                case 'address':
                    return this.renderAddressForm();
                case 'shipping':
                    return this.renderShippingOptions();
                case 'payment':
                    return this.renderPaymentOptions();
                default:
                    return this.renderAddressForm();
            }
        }
        
        renderAddressForm() {
            const saved = this.state.savedOrder || {};
            const savedDel = this.state.savedDelivery || {};
            
            return `
                <form id="checkout-address-form" class="space-y-4 sm:space-y-6">
                    <h2 class="font-serif text-lg sm:text-xl mb-4 sm:mb-6">Leveringsoplysninger</h2>
                    
                    ${saved.email ? '<div class="p-3 bg-green-50 border border-green-200 rounded mb-4 text-sm">Dine oplysninger er gemt fra tidligere</div>' : ''}
                    
                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                        <div>
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Email *</label>
                            <input type="email" name="email" required value="${saved.email || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                        <div>
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Telefon</label>
                            <input type="tel" name="phone" value="${saved.phone || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                        <div>
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Navn *</label>
                            <input type="text" name="name" required value="${saved.name || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                        <div>
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Firma</label>
                            <input type="text" name="company_name" value="${saved.company_name || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium mb-1 sm:mb-2">Adresse *</label>
                        <input type="text" name="address" required value="${saved.address || ''}"
                               class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                    </div>
                    
                    <div class="grid grid-cols-2 sm:grid-cols-3 gap-3 sm:gap-6">
                        <div class="col-span-2 sm:col-span-1">
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Postnr. *</label>
                            <input type="text" name="zipcode" required value="${saved.zipcode || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label class="block text-sm font-medium mb-1 sm:mb-2">By *</label>
                            <input type="text" name="city" required value="${saved.city || ''}"
                                   class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label class="block text-sm font-medium mb-1 sm:mb-2">Land *</label>
                            <select name="country_id" required 
                                    class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                ${(this.state.countries || [{id: 1, name: 'Danmark'}]).map(country => 
                                    `<option value="${country.id}" ${saved.country_id == country.id ? 'selected' : ''}>${country.name}</option>`
                                ).join('')}
                            </select>
                        </div>
                    </div>
                    
                    <div class="mt-6 sm:mt-8 p-3 sm:p-4 bg-gray-50 rounded">
                        <h3 class="font-medium mb-3 sm:mb-4">
                            <input type="checkbox" id="different-delivery" class="mr-2" onchange="document.getElementById('delivery-fields').style.display = this.checked ? 'block' : 'none'">
                            <label for="different-delivery">Levér til anden adresse</label>
                        </h3>
                        
                        <div id="delivery-fields" style="display: none" class="space-y-4 sm:space-y-6">
                            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                                <div>
                                    <label class="block text-sm font-medium mb-1 sm:mb-2">Leveringsnavn</label>
                                    <input type="text" name="delivery_name" 
                                           class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium mb-1 sm:mb-2">Leveringsfirma</label>
                                    <input type="text" name="delivery_company" 
                                           class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                </div>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium mb-1 sm:mb-2">Leveringsadresse</label>
                                <input type="text" name="delivery_address" 
                                       class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                            </div>
                            
                            <div class="grid grid-cols-2 sm:grid-cols-3 gap-3 sm:gap-6">
                                <div class="col-span-2 sm:col-span-1">
                                    <label class="block text-sm font-medium mb-1 sm:mb-2">Lev. postnr.</label>
                                    <input type="text" name="delivery_zipcode" 
                                           class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                </div>
                                <div class="col-span-2 sm:col-span-1">
                                    <label class="block text-sm font-medium mb-1 sm:mb-2">Lev. by</label>
                                    <input type="text" name="delivery_city" 
                                           class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                </div>
                                <div class="col-span-2 sm:col-span-1">
                                    <label class="block text-sm font-medium mb-1 sm:mb-2">Lev. land</label>
                                    <select name="delivery_country_id" 
                                            class="w-full px-3 sm:px-4 py-2 border border-light-gray focus:border-charcoal rounded-md">
                                        <option value="">Samme som fakturaadresse</option>
                                        ${(this.state.countries || [{id: 1, name: 'Danmark'}]).map(country => 
                                            `<option value="${country.id}">${country.name}</option>`
                                        ).join('')}
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="w-full sm:w-auto bg-charcoal text-white px-6 sm:px-8 py-3 hover:bg-nordic rounded-md transition-colors mt-6">
                        Fortsæt til forsendelse
                    </button>
                </form>
            `;
        }
        
        renderShippingOptions() {
            // Auto-select first shipping method if none selected
            if (!this.state.selectedShipping && this.state.shippingMethods?.length > 0) {
                this.state.selectedShipping = this.state.shippingMethods[0].id;
                // If first method has delivery shops, select first shop
                const firstMethod = this.state.shippingMethods[0];
                if (firstMethod.has_delivery_shops && firstMethod.delivery_shops?.length > 0) {
                    this.state.selectedDeliveryShop = firstMethod.delivery_shops[0].number;
                }
            }
            
            return `
                <div class="space-y-6">
                    <h2 class="font-serif text-xl mb-6">Vælg forsendelse</h2>
                    
                    ${(this.state.shippingMethods || []).map((method, index) => {
                        const isSelected = this.state.selectedShipping == method.id || (!this.state.selectedShipping && index === 0);
                        const methodId = `shipping_${method.id}`;
                        
                        return `
                        <div class="shipping-method-container ${isSelected ? 'border-charcoal' : 'border-light-gray'} border rounded-lg overflow-hidden transition-all">
                            <label class="flex items-center p-4 hover:bg-beige/20 cursor-pointer" for="${methodId}">
                                <input type="radio" 
                                       id="${methodId}"
                                       name="shipping" 
                                       value="${method.id}" 
                                       ${isSelected ? 'checked' : ''}
                                       onchange="window.washingtonApp.handleShippingChange('${method.id}')"
                                       class="mr-4">
                                <div class="flex-1">
                                    <div class="font-medium">${method.name}</div>
                                    ${method.description ? `<div class="text-sm text-charcoal/60">${method.description}</div>` : ''}
                                </div>
                                <div class="font-medium">${formatPrice(method.price)} ${window.shopConfig.currency}</div>
                            </label>
                            
                            ${method.has_delivery_shops && isSelected ? `
                                <div class="border-t border-light-gray bg-beige/10 p-4">
                                    ${method.delivery_shops && method.delivery_shops.length > 0 ? `
                                        <div class="mb-3">
                                            <h3 class="font-medium mb-2 flex items-center">
                                                <svg class="w-5 h-5 mr-2 text-charcoal/60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path>
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                                </svg>
                                                Vælg pakkeshop
                                            </h3>
                                            
                                            <!-- Search field for delivery shops -->
                                            <div class="mb-3">
                                                <input type="text" 
                                                       placeholder="Søg efter postnummer eller by..." 
                                                       onkeyup="window.washingtonApp.filterDeliveryShops(this.value, '${method.id}')"
                                                       class="w-full px-3 py-2 border border-light-gray rounded focus:border-charcoal text-sm">
                                            </div>
                                            
                                            <div class="delivery-shops-list space-y-2 border border-light-gray rounded p-2 bg-white" style="height: 200px; overflow-y: auto;" data-method-id="${method.id}">
                                                ${method.delivery_shops.map((shop, shopIndex) => `
                                                    <label class="delivery-shop-item flex items-start p-3 border border-light-gray hover:border-charcoal hover:bg-beige/20 cursor-pointer rounded transition-all"
                                                           data-shop-zip="${shop.zip}" 
                                                           data-shop-city="${shop.city?.toLowerCase()}"
                                                           data-shop-name="${shop.name?.toLowerCase()}"
                                                           data-shop-street="${shop.street?.toLowerCase()}">
                                                        <input type="radio" 
                                                               name="delivery_shop" 
                                                               value="${shop.number}" 
                                                               ${shopIndex === 0 && isSelected ? 'checked' : ''}
                                                               onchange="window.washingtonApp.state.selectedDeliveryShop = this.value; window.washingtonApp.updateSelectedShopDisplay('${shop.number}', '${method.id}')"
                                                               class="mr-4 mt-1 flex-shrink-0">
                                                        <div class="flex-1">
                                                            <div class="font-medium text-sm">${shop.name}</div>
                                                            <div class="text-xs text-charcoal/60">${shop.street}</div>
                                                            <div class="text-xs text-charcoal/60">${shop.zip} ${shop.city}</div>
                                                        </div>
                                                        ${shop.distance ? `
                                                            <div class="text-xs text-charcoal/40 ml-2">${shop.distance} km</div>
                                                        ` : ''}
                                                    </label>
                                                `).join('')}
                                            </div>
                                            
                                            <div class="selected-shop-display mt-3 p-3 bg-green-50 border border-green-200 rounded hidden">
                                                <div class="flex items-start">
                                                    <svg class="w-5 h-5 mr-2 text-green-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                    </svg>
                                                    <div class="flex-1">
                                                        <div class="font-medium text-sm text-green-800">Valgt pakkeshop:</div>
                                                        <div class="selected-shop-info text-xs text-green-700"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    ` : `
                                        <div class="p-4 bg-yellow-50 border border-yellow-200 rounded">
                                            <p class="text-sm text-yellow-800">
                                                <svg class="w-5 h-5 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                                                </svg>
                                                Ingen pakkeshops fundet for dit postnummer. Prøv at opdatere dine leveringsoplysninger.
                                            </p>
                                        </div>
                                    `}
                                </div>
                            ` : ''}
                        </div>
                    `;
                    }).join('')}
                    
                    <div class="flex space-x-4">
                        <button data-checkout-back="address" 
                                class="px-8 py-3 border border-charcoal text-charcoal hover:bg-light-gray">
                            Tilbage
                        </button>
                        <button data-checkout-next="payment" 
                                class="bg-charcoal text-white px-8 py-3 hover:bg-nordic">
                            Fortsæt til betaling
                        </button>
                    </div>
                </div>
            `;
        }
        
        renderPaymentOptions() {
            // Auto-select first payment method if none selected
            if (!this.state.selectedPayment && this.state.paymentMethods?.length > 0) {
                this.state.selectedPayment = this.state.paymentMethods[0].id;
            }
            
            return `
                <div class="space-y-6">
                    <h2 class="font-serif text-xl mb-6">Vælg betalingsmetode</h2>
                    
                    ${(this.state.paymentMethods || []).map((method, index) => `
                        <label class="flex items-center p-4 border border-light-gray hover:border-charcoal cursor-pointer">
                            <input type="radio" name="payment" value="${method.id}" 
                                   ${this.state.selectedPayment == method.id || (!this.state.selectedPayment && index === 0) ? 'checked' : ''}
                                   onchange="window.washingtonApp.state.selectedPayment = this.value"
                                   class="mr-4">
                            <div class="flex-1">
                                <div class="font-medium">${method.name}</div>
                            </div>
                            ${method.fee ? `<div class="font-medium">+${method.fee} ${window.shopConfig.currency}</div>` : ''}
                        </label>
                    `).join('')}
                    
                    <div class="my-6 p-6 border border-light-gray bg-white">
                        <label class="flex items-start cursor-pointer">
                            <input type="checkbox" id="accept-terms" 
                                   class="w-5 h-5 mr-4 mt-0.5 text-charcoal border-2 border-charcoal/30 rounded-sm focus:ring-2 focus:ring-charcoal/20 cursor-pointer"
                                   style="accent-color: #262626; margin-right: 1rem;"
                                   onchange="window.washingtonApp.state.termsAccepted = this.checked; window.washingtonApp.updateCompleteOrderButton()">
                            <span class="text-sm text-charcoal select-none">
                                Jeg accepterer 
                                <button type="button" 
                                        onclick="event.preventDefault(); window.washingtonApp.showTermsModal()" 
                                        class="font-semibold underline hover:no-underline">
                                    handelsbetingelserne
                                </button>
                                for at fortsætte med købet
                            </span>
                        </label>
                    </div>
                    
                    <div class="flex space-x-4">
                        <button data-checkout-back="shipping" 
                                class="px-8 py-3 border border-charcoal text-charcoal hover:bg-light-gray">
                            Tilbage
                        </button>
                        <button data-complete-order 
                                class="bg-charcoal text-white px-8 py-3 hover:bg-nordic disabled:opacity-50 disabled:cursor-not-allowed"
                                ${!this.state.termsAccepted ? 'disabled' : ''}>
                            Gennemfør ordre
                        </button>
                    </div>
                </div>
            `;
        }

        renderCartPage() {
            const items = this.state.cart.items || [];
            
            if (items.length === 0) {
                return `
                    <div class="px-6 lg:px-12 py-20 text-center">
                        <h2 class="font-serif text-3xl mb-4">Din kurv er tom</h2>
                        <p class="text-charcoal/60 mb-8">Tilføj produkter for at komme i gang</p>
                        <a href="/" data-route="/" 
                           class="inline-block bg-charcoal text-white px-8 py-3 hover:bg-nordic">
                            Fortsæt med at handle
                        </a>
                    </div>
                `;
            }
            
            return `
                <div class="px-6 lg:px-12 py-12">
                    <h1 class="font-serif text-3xl mb-8">Indkøbskurv</h1>
                    
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-12">
                        <div class="lg:col-span-2">
                            <ul class="space-y-4 sm:space-y-6">
                                ${items.map(item => `
                                    <li class="group">
                                        <div class="bg-white rounded-lg border border-light-gray hover:shadow-sm transition-shadow">
                                            <div class="flex gap-3 sm:gap-4 p-4 sm:p-6">
                                                <!-- Product Image -->
                                                <div class="flex-shrink-0">
                                                    <img src="${item.image?.formats?.thumbnail || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"%3E%3Crect width="80" height="80" fill="%23f5f5dc"/%3E%3C/svg%3E'}" 
                                                         class="w-16 h-16 sm:w-20 sm:h-20 rounded-md object-cover">
                                                </div>
                                                
                                                <!-- Product Details -->
                                                <div class="flex-1 min-w-0">
                                                    <!-- Top row: Product name + remove button on mobile -->
                                                    <div class="flex justify-between items-start mb-2 sm:mb-3">
                                                        <div class="min-w-0 flex-1 pr-2">
                                                            <h3 class="text-base sm:text-lg font-medium text-charcoal leading-tight">
                                                                <a href="#" class="hover:text-nordic transition-colors">${item.name}</a>
                                                            </h3>
                                                            ${this.formatVariantInfo(item)}
                                                        </div>
                                                        
                                                        <!-- Remove button (mobile: top right, desktop: with other actions) -->
                                                        <button data-remove-from-cart="${item.product_id}"
                                                                class="sm:hidden w-8 h-8 rounded flex items-center justify-center text-charcoal/40 hover:text-red-600 hover:bg-red-50 transition-all">
                                                            ×
                                                        </button>
                                                    </div>
                                                    
                                                    <!-- Bottom row: Controls and price -->
                                                    <div class="flex justify-between items-center">
                                                        <!-- Quantity Controls -->
                                                        <div class="flex items-center gap-1 sm:gap-2">
                                                            <button data-update-cart="${item.product_id}" data-amount="${item.amount - 1}"
                                                                    class="w-8 h-8 rounded border border-light-gray flex items-center justify-center hover:bg-light-gray transition-colors">
                                                                −
                                                            </button>
                                                            <span class="min-w-[2.5rem] px-3 text-center font-medium text-sm sm:text-base">${item.amount}</span>
                                                            <button data-update-cart="${item.product_id}" data-amount="${item.amount + 1}"
                                                                    class="w-8 h-8 rounded border border-light-gray flex items-center justify-center hover:bg-light-gray transition-colors">
                                                                +
                                                            </button>
                                                        </div>
                                                        
                                                        <!-- Price and desktop remove -->
                                                        <div class="flex items-center gap-4 sm:gap-6">
                                                            <div class="text-right">
                                                                <div class="text-base sm:text-lg font-semibold text-charcoal">${formatPrice(item.subtotal)} ${window.shopConfig.currency}</div>
                                                                <div class="text-xs sm:text-sm text-charcoal/60">${formatPrice(item.price)} ${window.shopConfig.currency} pr. stk</div>
                                                            </div>
                                                            
                                                            <!-- Remove button (desktop only) -->
                                                            <button data-remove-from-cart="${item.product_id}"
                                                                    class="hidden sm:flex w-8 h-8 rounded items-center justify-center text-charcoal/40 hover:text-red-600 hover:bg-red-50 transition-all">
                                                                ×
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </li>
                                `).join('')}
                            </ul>
                        </div>
                        
                        <div class="bg-beige p-8 h-fit">
                            <h2 class="font-serif text-xl mb-6">Ordreoversigt</h2>
                            
                            <!-- Voucher Code Section -->
                            <div class="mb-6 pb-6 border-b border-charcoal/10">
                                ${this.state.cart.voucher ? `
                                    <div class="p-3 bg-green-50 border border-green-200 rounded flex justify-between items-center mb-4">
                                        <span class="text-sm text-green-700">
                                            Rabatkode "${this.state.cart.voucher.code}" anvendt
                                        </span>
                                        <button id="remove-voucher" class="text-red-600 hover:text-red-800 text-sm">
                                            Fjern
                                        </button>
                                    </div>
                                ` : `
                                    <button id="toggle-voucher" class="text-sm text-charcoal/60 hover:text-charcoal flex items-center space-x-1">
                                        <span id="voucher-toggle-icon" class="text-xs mr-2">→</span>
                                        <span>Har du en rabatkode?</span>
                                    </button>
                                    <div id="voucher-form" class="hidden mt-3">
                                        <div class="flex space-x-2">
                                            <input type="text" 
                                                   id="voucher-code" 
                                                   placeholder="Indtast kode"
                                                   class="flex-1 px-3 py-2 border border-charcoal/20 focus:outline-none focus:border-charcoal">
                                            <button id="apply-voucher" 
                                                    class="px-4 py-2 bg-charcoal text-white hover:bg-nordic">
                                                Anvend
                                            </button>
                                        </div>
                                        <div id="voucher-message" class="mt-2 text-sm hidden"></div>
                                    </div>
                                `}
                            </div>
                            
                            <div class="space-y-3 mb-6 pb-6 border-b border-charcoal/10">
                                <div class="flex justify-between">
                                    <span>Subtotal</span>
                                    <span>${formatPrice(this.state.cart.subtotal)} ${window.shopConfig.currency}</span>
                                </div>
                                ${this.state.cart.campaign && this.state.cart.campaign.discount > 0 ? `
                                    <div class="flex justify-between text-green-600">
                                        <span>Kampagne rabat</span>
                                        <span>-${formatPrice(this.state.cart.campaign.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                ${this.state.cart.voucher && this.state.cart.voucher.discount > 0 ? `
                                    <div class="flex justify-between text-green-600">
                                        <span>Rabat</span>
                                        <span>-${formatPrice(this.state.cart.voucher.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                <div class="flex justify-between">
                                    <span>Levering</span>
                                    <span>${this.state.cart.shipping > 0 ? formatPrice(this.state.cart.shipping) + ' ' + window.shopConfig.currency : 'Beregnes ved kassen'}</span>
                                </div>
                            </div>
                            
                            <div class="flex justify-between text-xl font-medium mb-8">
                                <span>Total</span>
                                <span>${formatPrice(this.state.cart.total)} ${window.shopConfig.currency}</span>
                            </div>
                            
                            <button data-route="/address" class="w-full bg-charcoal text-white py-4 hover:bg-nordic">
                                GÅ TIL KASSEN
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }

        renderCartSidebar() {
            const items = this.state.cart.items || [];
            
            return `
                <div id="cart-sidebar" class="${this.state.cartOpen ? '' : 'translate-x-full'} 
                     fixed right-0 top-0 h-full bg-white shadow-2xl z-50 transition-transform duration-300 flex flex-col" style="width: ${window.innerWidth > 640 ? '384px' : '100%'}">
                    <div class="p-6 border-b border-light-gray flex justify-between items-center flex-shrink-0">
                        <h2 class="font-serif text-xl">Indkøbskurv</h2>
                        <button id="cart-close" class="text-charcoal/40 hover:text-charcoal">
                            <svg class="w-6 h-6 pointer-events-none" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                    
                    <div class="flex-1 overflow-y-auto p-6" style="min-height: 0;">
                        ${items.length === 0 ? `
                            <p class="text-center text-charcoal/60 py-8">Din kurv er tom</p>
                        ` : `
                            <div class="space-y-4">
                                ${items.map(item => `
                                    <div class="flex space-x-3">
                                        <img src="${item.image?.formats?.thumbnail || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"%3E%3Crect width="80" height="80" fill="%23f5f5dc"/%3E%3C/svg%3E'}" 
                                             class="w-20 h-20 object-contain p-1">
                                        <div class="flex-1">
                                            <h4 class="font-medium mb-1">${item.name}</h4>
                                            ${this.formatVariantInfo(item)}
                                            <p class="text-sm text-charcoal/60 mb-2">${formatPrice(item.price)} ${window.shopConfig.currency}</p>
                                            <div class="flex items-center space-x-2">
                                                <button data-update-cart="${item.product_id}" data-amount="${item.amount - 1}"
                                                        class="text-charcoal/40 hover:text-charcoal">-</button>
                                                <span class="px-2">${item.amount}</span>
                                                <button data-update-cart="${item.product_id}" data-amount="${item.amount + 1}"
                                                        class="text-charcoal/40 hover:text-charcoal">+</button>
                                            </div>
                                        </div>
                                        <button data-remove-from-cart="${item.product_id}"
                                                class="text-charcoal/40 hover:text-charcoal">
                                            <svg class="w-5 h-5 pointer-events-none" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                        </button>
                                    </div>
                                `).join('')}
                            </div>
                        `}
                    </div>
                    
                    ${items.length > 0 ? `
                        <div class="border-t border-light-gray p-6 flex-shrink-0">
                            <div class="space-y-2 mb-4">
                                <div class="flex justify-between text-sm">
                                    <span>Subtotal</span>
                                    <span>${formatPrice(this.state.cart.subtotal || 0)} ${window.shopConfig.currency}</span>
                                </div>
                                ${this.state.cart.campaign && this.state.cart.campaign.discount > 0 ? `
                                    <div class="flex justify-between text-sm text-green-600">
                                        <span>Kampagne</span>
                                        <span>-${formatPrice(this.state.cart.campaign.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                ${this.state.cart.voucher && this.state.cart.voucher.discount > 0 ? `
                                    <div class="flex justify-between text-sm text-green-600">
                                        <span>Rabat</span>
                                        <span>-${formatPrice(this.state.cart.voucher.discount)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                ${this.state.cart.shipping > 0 ? `
                                    <div class="flex justify-between text-sm">
                                        <span>Fragt</span>
                                        <span>${formatPrice(this.state.cart.shipping)} ${window.shopConfig.currency}</span>
                                    </div>
                                ` : ''}
                                <div class="flex justify-between text-lg font-medium pt-2 border-t">
                                    <span>Total</span>
                                    <span>${formatPrice(this.state.cart.total || 0)} ${window.shopConfig.currency}</span>
                                </div>
                            </div>
                            <a href="/basket" data-route="/basket" 
                               class="block w-full bg-charcoal text-white py-3 text-center hover:bg-nordic">
                                SE KURV
                            </a>
                        </div>
                    ` : ''}
                </div>
                
                <!-- Backdrop -->
                <div id="cart-backdrop" class="${this.state.cartOpen ? '' : 'hidden'} 
                     fixed inset-0 bg-black/30 z-40"></div>
            `;
        }

        renderSearchOverlay() {
            const searchResults = this.state.searchResults || [];
            const isSearching = this.state.isSearching || false;
            
            return `
                <div id="search-overlay" class="${this.state.searchOpen ? '' : 'hidden'} 
                     fixed inset-0 bg-white z-50 overflow-y-auto">
                    <div class="px-6 lg:px-12 pt-20">
                        <div class="flex items-center space-x-4">
                            <input type="text" 
                                   id="search-input"
                                   placeholder="Søg efter produkter..."
                                   class="flex-1 text-2xl font-light border-b-2 border-charcoal/20 pb-2 focus:outline-none focus:border-charcoal"
                                   value="${this.state.searchQuery || ''}"
                                   autofocus>
                            <button id="search-close" class="text-charcoal/40 hover:text-charcoal">
                                <svg class="w-8 h-8 pointer-events-none" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>
                        
                        <div id="search-results" class="mt-8">
                            ${isSearching ? `
                                <p class="text-charcoal/40">Søger...</p>
                            ` : searchResults.length > 0 ? `
                                <div class="grid grid-cols-4 md:grid-cols-6 lg:grid-cols-8 gap-4">
                                    ${searchResults.map(product => {
                                        const productUrl = product.rewrite_url || `product/${product.id}`;
                                        const imageUrl = product.images?.[0]?.formats?.large || 
                                                       product.images?.[0]?.url || 
                                                       'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="400" height="400" viewBox="0 0 400 400"%3E%3Crect width="400" height="400" fill="%23f5f5dc"/%3E%3C/svg%3E';
                                        return `
                                            <a href="/${productUrl}" data-route="/${productUrl}" 
                                               class="group cursor-pointer">
                                                <div class="aspect-square overflow-hidden bg-light-gray">
                                                    <img src="${imageUrl}" 
                                                         alt="${product.name}"
                                                         class="w-full h-full object-contain p-4 group-hover:scale-105 transition-transform duration-300"
                                                         loading="lazy">
                                                </div>
                                                <div class="mt-3">
                                                    <h3 class="text-base font-medium group-hover:text-nordic line-clamp-2">${product.name}</h3>
                                                    <p class="text-base font-semibold mt-1">${formatPrice(product.price)} ${window.shopConfig.currency}</p>
                                                </div>
                                            </a>
                                        `;
                                    }).join('')}
                                </div>
                            ` : this.state.searchQuery ? `
                                <p class="text-charcoal/40">Ingen produkter fundet</p>
                            ` : `
                                <p class="text-charcoal/40">Begynd at skrive for at søge...</p>
                            `}
                        </div>
                    </div>
                </div>
            `;
        }

        renderFooter() {
            return `
                <footer class="border-t py-4 mt-12">
                    <div class="px-6 lg:px-12 text-center text-sm text-gray-500">
                        &copy; 2024 ${window.shopConfig.shopName}
                    </div>
                </footer>
            `;
        }

        renderLoading() {
            return `
                <div class="flex justify-center items-center py-20">
                    <div class="relative">
                        <div class="w-16 h-16 border-4 border-sage/20 rounded-full"></div>
                        <div class="w-16 h-16 border-4 border-sage border-t-transparent rounded-full animate-spin absolute top-0"></div>
                    </div>
                </div>
            `;
        }

        attachEventListeners() {
            // Global keyboard listeners
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape') {
                    if (this.state.searchOpen) {
                        this.state.searchOpen = false;
                        this.state.searchResults = [];
                        this.state.searchQuery = '';
                        this.render();
                    } else if (this.state.cartOpen) {
                        this.state.cartOpen = false;
                        this.render();
                    } else if (this.state.mobileMenuOpen) {
                        this.state.mobileMenuOpen = false;
                        this.render();
                    }
                }
            });
            
            // Mobile Menu Toggle
            const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
            if (mobileMenuToggle) {
                mobileMenuToggle.addEventListener('click', () => {
                    this.state.mobileMenuOpen = !this.state.mobileMenuOpen;
                    this.render();
                });
            }
            
            // Mobile Menu Close
            const mobileMenuClose = document.getElementById('mobile-menu-close');
            if (mobileMenuClose) {
                mobileMenuClose.addEventListener('click', () => {
                    this.state.mobileMenuOpen = false;
                    this.render();
                });
            }
            
            // Mobile Menu Backdrop Click
            const mobileMenu = document.getElementById('mobile-menu');
            if (mobileMenu) {
                mobileMenu.addEventListener('click', (e) => {
                    if (e.target === mobileMenu) {
                        this.state.mobileMenuOpen = false;
                        this.render();
                    }
                });
            }
            
            
            // Close mobile menu when clicking on a link - using event delegation
            document.addEventListener('click', (e) => {
                if (e.target.closest('.mobile-menu-link')) {
                    this.state.mobileMenuOpen = false;
                    this.render();
                }
            });
            
            // Desktop submenu hover handling
            document.querySelectorAll('.group').forEach(menuItem => {
                const submenu = menuItem.querySelector('.submenu-dropdown');
                if (submenu) {
                    menuItem.addEventListener('mouseenter', () => {
                        submenu.style.opacity = '1';
                        submenu.style.visibility = 'visible';
                    });
                    menuItem.addEventListener('mouseleave', () => {
                        submenu.style.opacity = '0';
                        submenu.style.visibility = 'hidden';
                    });
                }
            });
            
            // Search
            const searchToggle = document.getElementById('search-toggle');
            const searchClose = document.getElementById('search-close');
            const searchInput = document.getElementById('search-input');
            
            if (searchToggle) {
                searchToggle.addEventListener('click', () => {
                    this.state.searchOpen = !this.state.searchOpen;
                    this.render();
                    if (this.state.searchOpen) {
                        // Use setTimeout to ensure DOM is updated before focusing
                        setTimeout(() => {
                            const input = document.getElementById('search-input');
                            if (input) {
                                input.focus();
                            }
                        }, 50);
                    }
                });
            }
            
            if (searchClose) {
                searchClose.addEventListener('click', () => {
                    this.state.searchOpen = false;
                    this.render();
                });
            }
            
            if (searchInput) {
                // Live search with debounce
                let searchTimeout;
                searchInput.addEventListener('input', (e) => {
                    const query = e.target.value;
                    
                    // Clear previous timeout
                    clearTimeout(searchTimeout);
                    
                    // If query is empty, don't search
                    if (!query.trim()) {
                        return;
                    }
                    
                    // Debounce search by 300ms
                    searchTimeout = setTimeout(() => {
                        this.search(query);
                        // Don't close overlay on live search
                    }, 300);
                });
                
                // Handle Enter and Escape keys
                searchInput.addEventListener('keydown', (e) => {
                    if (e.key === 'Enter') {
                        clearTimeout(searchTimeout);
                        this.search(e.target.value);
                        this.state.searchOpen = false;
                    } else if (e.key === 'Escape') {
                        this.state.searchOpen = false;
                        this.state.searchResults = [];
                        this.state.searchQuery = '';
                        this.render();
                    }
                });
            }
            
            // Cart
            const cartToggle = document.getElementById('cart-toggle');
            const cartClose = document.getElementById('cart-close');
            
            if (cartToggle) {
                cartToggle.addEventListener('click', () => {
                    this.state.cartOpen = !this.state.cartOpen;
                    this.render();
                });
            }
            
            if (cartClose) {
                cartClose.addEventListener('click', () => {
                    this.state.cartOpen = false;
                    this.render();
                });
            }
            
            // Cart backdrop
            const cartBackdrop = document.getElementById('cart-backdrop');
            if (cartBackdrop) {
                cartBackdrop.addEventListener('click', () => {
                    this.state.cartOpen = false;
                    this.render();
                });
            }
            
            // Voucher code handlers
            const applyVoucherBtn = document.getElementById('apply-voucher');
            const removeVoucherBtn = document.getElementById('remove-voucher');
            const voucherInput = document.getElementById('voucher-code');
            
            if (applyVoucherBtn) {
                applyVoucherBtn.addEventListener('click', async () => {
                    if (!voucherInput) return;
                    
                    const code = voucherInput.value.trim();
                    if (!code) {
                        this.showVoucherMessage('Indtast venligst en rabatkode', 'error');
                        return;
                    }
                    
                    // Create a form and submit it to apply voucher
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = window.location.href;
                    
                    const voucherField = document.createElement('input');
                    voucherField.type = 'hidden';
                    voucherField.name = 'voucher';
                    voucherField.value = code;
                    form.appendChild(voucherField);
                    
                    const updateField = document.createElement('input');
                    updateField.type = 'hidden';
                    updateField.name = 'update';
                    updateField.value = 'Redeem';
                    form.appendChild(updateField);
                    
                    document.body.appendChild(form);
                    form.submit();
                });
            }
            
            if (removeVoucherBtn) {
                removeVoucherBtn.addEventListener('click', async () => {
                    // Create a form to remove voucher
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = window.location.href;
                    
                    const voucherField = document.createElement('input');
                    voucherField.type = 'hidden';
                    voucherField.name = 'voucher';
                    voucherField.value = '';
                    form.appendChild(voucherField);
                    
                    document.body.appendChild(form);
                    form.submit();
                });
            }
            
            // Enter key support for voucher input
            if (voucherInput) {
                voucherInput.addEventListener('keypress', (e) => {
                    if (e.key === 'Enter' && applyVoucherBtn) {
                        applyVoucherBtn.click();
                    }
                });
            }
            
            // Voucher toggle functionality
            const toggleVoucherBtn = document.getElementById('toggle-voucher');
            const voucherForm = document.getElementById('voucher-form');
            const voucherToggleIcon = document.getElementById('voucher-toggle-icon');
            
            if (toggleVoucherBtn && voucherForm && voucherToggleIcon) {
                toggleVoucherBtn.addEventListener('click', () => {
                    const isHidden = voucherForm.classList.contains('hidden');
                    
                    if (isHidden) {
                        voucherForm.classList.remove('hidden');
                        voucherToggleIcon.textContent = '↓';
                    } else {
                        voucherForm.classList.add('hidden');
                        voucherToggleIcon.textContent = '→';
                    }
                });
            }

            // Quantity input handling
            document.querySelectorAll('.quantity-input').forEach(input => {
                input.addEventListener('change', (e) => {
                    const productId = e.target.getAttribute('data-product-id');
                    const newAmount = parseInt(e.target.value) || 1;
                    
                    if (newAmount < 1) {
                        e.target.value = 1;
                        return;
                    }
                    
                    this.updateCart(productId, newAmount);
                });
                
                input.addEventListener('blur', (e) => {
                    if (!e.target.value || e.target.value < 1) {
                        e.target.value = 1;
                    }
                });
            });
            
            // Add to cart
            document.querySelectorAll('[data-add-to-cart]').forEach(btn => {
                btn.addEventListener('click', (e) => {
                    const productId = e.currentTarget.getAttribute('data-add-to-cart');
                    
                    // Check if we're on a product page with variants
                    const attributes = {};
                    let allSelected = true;
                    
                    // Check for select dropdowns
                    const variantSelects = document.querySelectorAll('select[data-variant-attribute]');
                    variantSelects.forEach(select => {
                        const attrId = select.getAttribute('data-variant-attribute');
                        const value = select.value;
                        
                        if (!value) {
                            allSelected = false;
                            select.classList.add('border-red-500');
                        } else {
                            select.classList.remove('border-red-500');
                            attributes[attrId] = value;
                        }
                    });
                    
                    // Check for button variant groups
                    const variantGroups = document.querySelectorAll('[data-variant-group]');
                    variantGroups.forEach(group => {
                        const attrId = group.getAttribute('data-variant-group');
                        const selectedButton = group.querySelector('button[data-selected="true"]');
                        
                        if (!selectedButton) {
                            allSelected = false;
                            // Highlight the group as needing selection
                            group.classList.add('ring-2', 'ring-red-500', 'rounded-lg');
                        } else {
                            group.classList.remove('ring-2', 'ring-red-500', 'rounded-lg');
                            const valueId = selectedButton.getAttribute('data-variant-value');
                            attributes[attrId] = valueId;
                        }
                    });
                    
                    if ((variantSelects.length > 0 || variantGroups.length > 0) && !allSelected) {
                        this.showNotification('Vælg venligst alle varianter før du tilføjer til kurv', 'error');
                        return;
                    }
                    
                    this.addToCart(productId, 1, attributes);
                });
            });
            
            // Variant select change handler (for dropdowns)
            document.querySelectorAll('select[data-variant-attribute]').forEach(select => {
                select.addEventListener('change', (e) => {
                    // Update variant info if needed
                    this.updateVariantInfo();
                });
            });
            
            // Variant button click handler (for fancy selector)
            document.querySelectorAll('button.variant-option').forEach(button => {
                button.addEventListener('click', (e) => {
                    if (button.disabled) return;
                    
                    const attrId = button.getAttribute('data-variant-attribute');
                    const valueId = button.getAttribute('data-variant-value');
                    const group = button.parentElement;
                    
                    // Remove selected state from all buttons in this group
                    group.querySelectorAll('button').forEach(btn => {
                        btn.classList.remove('border-nordic', 'bg-nordic/10');
                        btn.classList.add('border-charcoal/20');
                    });
                    
                    // Add selected state to clicked button
                    button.classList.remove('border-charcoal/20');
                    button.classList.add('border-nordic', 'bg-nordic/10');
                    
                    // Store the selection
                    button.setAttribute('data-selected', 'true');
                    
                    // Update variant info
                    this.updateVariantInfo();
                });
            });
            
            // Cart event listeners
            this.attachCartEventListeners();
            
            // Checkout form
            const checkoutForm = document.getElementById('checkout-address-form');
            if (checkoutForm) {
                checkoutForm.addEventListener('submit', async (e) => {
                    e.preventDefault();
                    const formData = new FormData(checkoutForm);
                    const data = Object.fromEntries(formData);
                    
                    try {
                        const response = await fetch(`${this.api.baseUrl}/checkout`, {
                            method: 'POST',
                            credentials: 'include',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ ...data, step: 'address' })
                        });
                        
                        if (response.ok) {
                            // Navigate to shipping step using correct URL
                            this.navigate('/shipping');
                        }
                    } catch (error) {
                        console.error('Checkout error:', error);
                    }
                });
            }
            
            // Checkout navigation
            document.querySelectorAll('[data-checkout-next]').forEach(btn => {
                btn.addEventListener('click', async () => {
                    const nextStep = btn.getAttribute('data-checkout-next');
                    
                    // Save current step data before moving to next
                    if (this.state.checkoutStep === 'shipping' && nextStep === 'payment') {
                        const shippingId = document.querySelector('input[name="shipping"]:checked')?.value;
                        const deliveryShopId = document.querySelector('input[name="delivery_shop"]:checked')?.value;
                        
                        if (shippingId) {
                            // Save shipping selection with delivery shop if selected
                            const shippingData = { 
                                step: 'shipping',
                                shipping_id: shippingId
                            };
                            
                            if (deliveryShopId) {
                                shippingData.delivery_shop_id = deliveryShopId;
                            }
                            
                            await fetch(`${this.api.baseUrl}/checkout`, {
                                method: 'POST',
                                credentials: 'include',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify(shippingData)
                            });
                            this.state.selectedShipping = shippingId;
                            this.state.selectedDeliveryShop = deliveryShopId;
                        }
                    }
                    
                    // Navigate to next step - stay on /shipping for payment to avoid conflicts
                    if (nextStep === 'payment') {
                        this.state.checkoutStep = 'payment';
                        this.render();
                        // Stay on /shipping URL to avoid /payment conflict
                        window.history.replaceState({}, '', '/shipping');
                    } else {
                        this.navigate(`/${nextStep}`);
                    }
                });
            });
            
            document.querySelectorAll('[data-checkout-back]').forEach(btn => {
                btn.addEventListener('click', () => {
                    const prevStep = btn.getAttribute('data-checkout-back');
                    // Navigate to previous step using correct URLs
                    this.navigate(`/${prevStep}`);
                });
            });
            
            // Complete order
            const completeBtn = document.querySelector('[data-complete-order]');
            if (completeBtn) {
                completeBtn.addEventListener('click', async () => {
                    // Check if terms are accepted
                    if (!this.state.termsAccepted) {
                        this.showNotification('Du skal acceptere handelsbetingelserne for at fortsætte', 'error');
                        return;
                    }
                    
                    // Use state values which are updated when selections change
                    const shippingId = this.state.selectedShipping;
                    const paymentId = this.state.selectedPayment || document.querySelector('input[name="payment"]:checked')?.value;
                    
                    console.log('Shipping ID from state:', shippingId);
                    console.log('Payment ID from state/form:', paymentId);
                    
                    if (!shippingId || !paymentId) {
                        if (!shippingId) {
                            this.showNotification('Vælg venligst leveringsmetode', 'error');
                            this.state.checkoutStep = 'shipping';
                            this.render();
                        } else {
                            this.showNotification('Vælg venligst betalingsmetode', 'error');
                        }
                        return;
                    }
                    
                    try {
                        // First save the shipping method with delivery shop if selected
                        const shippingData = { 
                            step: 'shipping',
                            shipping_id: shippingId
                        };
                        
                        // Add delivery shop if selected
                        if (this.state.selectedDeliveryShop) {
                            shippingData.delivery_shop_id = this.state.selectedDeliveryShop;
                        }
                        
                        const shippingResponse = await fetch(`${this.api.baseUrl}/checkout`, {
                            method: 'POST',
                            credentials: 'include',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify(shippingData)
                        });
                        
                        if (!shippingResponse.ok) {
                            console.error('Shipping step failed:', shippingResponse.status);
                            const errorText = await shippingResponse.text();
                            console.error('Shipping error:', errorText);
                            throw new Error('Failed to save shipping method');
                        }
                        
                        // Then save the payment method selection
                        const paymentResponse = await fetch(`${this.api.baseUrl}/checkout`, {
                            method: 'POST',
                            credentials: 'include',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ 
                                step: 'payment',
                                payment_gateway_id: paymentId
                            })
                        });
                        
                        if (!paymentResponse.ok) {
                            console.error('Payment step failed:', paymentResponse.status);
                            const errorText = await paymentResponse.text();
                            console.error('Payment error:', errorText);
                            throw new Error('Failed to save payment method');
                        }
                        
                        // Finally complete the order
                        const response = await fetch(`${this.api.baseUrl}/checkout`, {
                            method: 'POST',
                            credentials: 'include',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ 
                                step: 'complete',
                                payment_id: paymentId
                            })
                        });
                        
                        if (!response.ok) {
                            console.error('Complete order failed with status:', response.status);
                            const errorText = await response.text();
                            console.error('Complete order error response:', errorText);
                            throw new Error(`Failed to complete order: HTTP ${response.status}`);
                        }
                        
                        let result;
                        const responseText = await response.text();
                        try {
                            result = JSON.parse(responseText);
                        } catch (e) {
                            console.error('Failed to parse JSON response:', e);
                            console.error('Response text:', responseText);
                            throw new Error('Invalid response from server');
                        }
                        if (result.success) {
                            if (result.data.redirect) {
                                // Direct redirect to thanks page
                                window.location.href = result.data.redirect;
                            } else if (result.data.payment_redirect) {
                                // Store order info in session storage for recovery
                                sessionStorage.setItem('pending_order', JSON.stringify({
                                    basket_id: result.data.basket_id,
                                    payment_url: result.data.payment_url,
                                    timestamp: Date.now()
                                }));
                                
                                // Redirect to payment gateway
                                window.location.href = result.data.payment_url;
                            }
                        } else {
                            this.showNotification(result.error?.message || 'Der opstod en fejl', 'error');
                        }
                    } catch (error) {
                        console.error('Order completion error:', error);
                        this.showNotification('Der opstod en fejl ved ordreafslutning', 'error');
                    }
                });
            }
        }

        handleShippingChange(shippingId) {
            // Update selected shipping
            this.state.selectedShipping = shippingId;
            
            // Find the selected shipping method
            const method = this.state.shippingMethods.find(m => m.id === shippingId);
            if (method && method.has_delivery_shops && method.delivery_shops?.length > 0) {
                // Auto-select first delivery shop
                this.state.selectedDeliveryShop = method.delivery_shops[0].number;
            } else {
                // Clear delivery shop selection if not needed
                this.state.selectedDeliveryShop = null;
            }
            
            // Re-render to show/hide delivery shops
            this.render();
        }
        
        filterDeliveryShops(searchTerm, methodId) {
            const container = document.querySelector(`.delivery-shops-list[data-method-id="${methodId}"]`);
            if (!container) return;
            
            const searchLower = searchTerm.toLowerCase().trim();
            const shops = container.querySelectorAll('.delivery-shop-item');
            
            let visibleCount = 0;
            shops.forEach(shop => {
                const zip = shop.getAttribute('data-shop-zip');
                const city = shop.getAttribute('data-shop-city');
                const name = shop.getAttribute('data-shop-name');
                const street = shop.getAttribute('data-shop-street');
                
                if (!searchTerm || 
                    zip?.includes(searchLower) || 
                    city?.includes(searchLower) || 
                    name?.includes(searchLower) ||
                    street?.includes(searchLower)) {
                    shop.style.display = 'flex';
                    visibleCount++;
                } else {
                    shop.style.display = 'none';
                }
            });
            
            // Show message if no shops match
            let noResultsMsg = container.querySelector('.no-search-results');
            if (visibleCount === 0 && searchTerm) {
                if (!noResultsMsg) {
                    noResultsMsg = document.createElement('div');
                    noResultsMsg.className = 'no-search-results p-4 text-sm text-charcoal/60 text-center';
                    noResultsMsg.textContent = 'Ingen pakkeshops matcher din søgning';
                    container.appendChild(noResultsMsg);
                }
                noResultsMsg.style.display = 'block';
            } else if (noResultsMsg) {
                noResultsMsg.style.display = 'none';
            }
        }
        
        updateSelectedShopDisplay(shopNumber, methodId) {
            const method = this.state.shippingMethods.find(m => m.id === methodId);
            if (!method || !method.delivery_shops) return;
            
            const shop = method.delivery_shops.find(s => s.number === shopNumber);
            if (!shop) return;
            
            // Update display in the current method container
            const container = document.querySelector(`.shipping-method-container input[value="${methodId}"]`)?.closest('.shipping-method-container');
            if (container) {
                const display = container.querySelector('.selected-shop-display');
                const info = container.querySelector('.selected-shop-info');
                
                if (display && info) {
                    display.classList.remove('hidden');
                    info.innerHTML = `${shop.name}<br>${shop.street}, ${shop.zip} ${shop.city}`;
                }
            }
        }
        
        attachGlobalListeners() {
            // Scroll effect for header
            window.addEventListener('scroll', () => {
                const header = document.querySelector('header');
                if (header) {
                    if (window.scrollY > 50) {
                        header.classList.add('bg-white', 'shadow-sm');
                        header.classList.remove('bg-transparent');
                    } else {
                        header.classList.remove('bg-white', 'shadow-sm');
                        header.classList.add('bg-transparent');
                    }
                }
            });
        }
    }

    // Initialize app
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            window.washingtonApp = new WashingtonApp();
        });
    } else {
        window.washingtonApp = new WashingtonApp();
    }
})();